"""
Playlist Service for PurpleTube Player.
This module provides services for managing playlists, including creation,
retrieval, modification, and deletion of playlists and their videos.
"""
import logging
from typing import List, Optional
from datetime import datetime
from config.logging_config import logger
from config.settings import DBConfig
from core.database import db_manager
from core.models import Playlist, Video

class PlaylistService:
    """
    Service class for managing playlists and their videos.
    Provides methods for creating, retrieving, updating, and deleting playlists,
    as well as adding and removing videos from playlists.
    """

    @staticmethod
    def get_playlists() -> List[Playlist]:
        """
        Get all playlists from the database.
        Returns:
            List[Playlist]: A list of all playlists with their videos.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("SELECT id, name FROM playlists")
                playlists_data = cursor.fetchall()
                playlists = []
                for playlist_id, name in playlists_data:
                    cursor.execute("""
                        SELECT video_id, title, url, thumbnail_url, duration,
                               platform, author, published
                        FROM playlist_videos
                        WHERE playlist_id = ?
                        ORDER BY position ASC
                    """, (playlist_id,))
                    videos = []
                    for row in cursor.fetchall():
                        try:
                            video = Video(
                                title=row[1] or "No Title",
                                url=row[2] or "",
                                thumbnail_url=row[3] or "",
                                duration=row[4] or "0",
                                video_id=row[0],
                                platform=row[5] or "PeerTube",
                                author=row[6] or "Unknown",
                                published=datetime.fromisoformat(row[7]) if row[7] else None
                            )
                            videos.append(video)
                        except Exception as e:
                            logger.error(f"Error parsing video data: {e}")
                    playlists.append(Playlist(name=name, videos=videos, id=playlist_id))
                return playlists
        except Exception as e:
            logger.error(f"Error retrieving playlists: {e}")
            return []

    @staticmethod
    def get_playlist(playlist_id: int) -> Optional[Playlist]:
        """
        Get a playlist by its ID.
        Args:
            playlist_id: The ID of the playlist.
        Returns:
            Optional[Playlist]: The playlist if found, None otherwise.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("SELECT name FROM playlists WHERE id = ?", (playlist_id,))
                row = cursor.fetchone()
                if not row:
                    return None
                name = row[0]
                cursor.execute("""
                    SELECT video_id, title, url, thumbnail_url, duration,
                           platform, author, published
                    FROM playlist_videos
                    WHERE playlist_id = ?
                    ORDER BY position ASC
                """, (playlist_id,))
                videos = []
                for row in cursor.fetchall():
                    try:
                        video = Video(
                            title=row[1] or "No Title",
                            url=row[2] or "",
                            thumbnail_url=row[3] or "",
                            duration=row[4] or "0",
                            video_id=row[0],
                            platform=row[5] or "PeerTube",
                            author=row[6] or "Unknown",
                            published=datetime.fromisoformat(row[7]) if row[7] else None
                        )
                        videos.append(video)
                    except Exception as e:
                        logger.error(f"Error parsing video data: {e}")
                return Playlist(name=name, videos=videos, id=playlist_id)
        except Exception as e:
            logger.error(f"Error retrieving playlist: {e}")
            return None

    @staticmethod
    def get_watch_later_playlist() -> Optional[Playlist]:
        """
        Get the 'Watch Later' playlist.
        Returns:
            Optional[Playlist]: The 'Watch Later' playlist if it exists, None otherwise.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("SELECT id FROM playlists WHERE name = 'Watch Later'")
                row = cursor.fetchone()
                if not row:
                    return None
                playlist_id = row[0]
                cursor.execute("""
                    SELECT video_id, title, url, thumbnail_url, duration,
                           platform, author, published
                    FROM playlist_videos
                    WHERE playlist_id = ?
                    ORDER BY position ASC
                """, (playlist_id,))
                videos = []
                for row in cursor.fetchall():
                    try:
                        video = Video(
                            title=row[1] or "No Title",
                            url=row[2] or "",
                            thumbnail_url=row[3] or "",
                            duration=row[4] or "0",
                            video_id=row[0],
                            platform=row[5] or "PeerTube",
                            author=row[6] or "Unknown",
                            published=datetime.fromisoformat(row[7]) if row[7] else None
                        )
                        videos.append(video)
                    except Exception as e:
                        logger.error(f"Error parsing video data: {e}")
                return Playlist(name="Watch Later", videos=videos, id=playlist_id)
        except Exception as e:
            logger.error(f"Error retrieving 'Watch Later' playlist: {e}")
            return None

    @staticmethod
    def create_playlist(name: str) -> Optional[Playlist]:
        """
        Create a new playlist.
        Args:
            name: The name of the new playlist.
        Returns:
            Optional[Playlist]: The created playlist if successful, None otherwise.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("SELECT id FROM playlists WHERE name = ?", (name,))
                if cursor.fetchone():
                    logger.warning(f"Playlist with name '{name}' already exists")
                    return None
                cursor.execute("INSERT INTO playlists (name) VALUES (?)", (name,))
                playlist_id = cursor.lastrowid
                conn.commit()
                return Playlist(name=name, id=playlist_id)
        except Exception as e:
            logger.error(f"Error creating playlist: {e}")
            return None

    @staticmethod
    def save_playlist(playlist: Playlist) -> int:
        """
        Save a playlist to the database.
        Args:
            playlist: The playlist to save.
        Returns:
            int: The ID of the saved playlist.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("SELECT id FROM playlists WHERE name = ?", (playlist.name,))
                existing = cursor.fetchone()
                if existing:
                    playlist.id = existing[0]
                    return playlist.id
                cursor.execute("INSERT INTO playlists (name) VALUES (?)", (playlist.name,))
                playlist.id = cursor.lastrowid
                conn.commit()
                return playlist.id
        except Exception as e:
            logger.error(f"Error saving playlist: {e}")
            return -1

    @staticmethod
    def add_to_playlist(playlist_id: int, video: Video) -> bool:
        """
        Add a video to a playlist.
        Args:
            playlist_id: The ID of the playlist.
            video: The video to add.
        Returns:
            bool: True if the video was added successfully, False otherwise.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute(
                    "SELECT 1 FROM playlist_videos WHERE playlist_id = ? AND video_id = ?",
                    (playlist_id, video.video_id)
                )
                if cursor.fetchone():
                    logger.warning(f"Video {video.video_id} already in playlist {playlist_id}")
                    return False
                cursor.execute(
                    "SELECT COALESCE(MAX(position), 0) FROM playlist_videos WHERE playlist_id = ?",
                    (playlist_id,)
                )
                position = cursor.fetchone()[0] + 1
                cursor.execute("""
                    INSERT INTO playlist_videos
                    (playlist_id, video_id, position, title, url, thumbnail_url,
                     duration, platform, author, published)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                """, (
                    playlist_id,
                    video.video_id,
                    position,
                    video.title,
                    video.url,
                    video.thumbnail_url,
                    video.duration,
                    video.platform,
                    video.author,
                    video.published.isoformat() if video.published else None
                ))
                cursor.execute("""
                    UPDATE playlists
                    SET updated_at = CURRENT_TIMESTAMP
                    WHERE id = ?
                """, (playlist_id,))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Error adding video to playlist: {e}")
            return False

    @staticmethod
    def remove_from_playlist(playlist_id: int, video_id: str) -> bool:
        """
        Remove a video from a playlist.
        Args:
            playlist_id: The ID of the playlist.
            video_id: The ID of the video to remove.
        Returns:
            bool: True if the video was removed successfully, False otherwise.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute(
                    "SELECT position FROM playlist_videos WHERE playlist_id = ? AND video_id = ?",
                    (playlist_id, video_id)
                )
                position_row = cursor.fetchone()
                if not position_row:
                    logger.warning(f"Video {video_id} not found in playlist {playlist_id}")
                    return False
                position = position_row[0]
                cursor.execute(
                    "DELETE FROM playlist_videos WHERE playlist_id = ? AND video_id = ?",
                    (playlist_id, video_id)
                )
                cursor.execute("""
                    UPDATE playlist_videos
                    SET position = position - 1
                    WHERE playlist_id = ? AND position > ?
                """, (playlist_id, position))
                cursor.execute("""
                    UPDATE playlists
                    SET updated_at = CURRENT_TIMESTAMP
                    WHERE id = ?
                """, (playlist_id,))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Error removing video from playlist: {e}")
            return False

    @staticmethod
    def update_playlist_name(playlist_id: int, new_name: str) -> bool:
        """
        Update the name of a playlist.
        Args:
            playlist_id: The ID of the playlist.
            new_name: The new name for the playlist.
        Returns:
            bool: True if the playlist name was updated successfully, False otherwise.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute(
                    "SELECT 1 FROM playlists WHERE name = ? AND id != ?",
                    (new_name, playlist_id)
                )
                if cursor.fetchone():
                    logger.warning(f"Playlist with name '{new_name}' already exists")
                    return False
                cursor.execute("""
                    UPDATE playlists
                    SET name = ?, updated_at = CURRENT_TIMESTAMP
                    WHERE id = ?
                """, (new_name, playlist_id))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Error updating playlist name: {e}")
            return False

    @staticmethod
    def remove_playlist(playlist_id: int) -> bool:
        """
        Remove a playlist from the database.
        Args:
            playlist_id: The ID of the playlist to remove.
        Returns:
            bool: True if the playlist was removed successfully, False otherwise.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute(
                    "SELECT 1 FROM playlists WHERE id = ? AND name = 'Watch Later'",
                    (playlist_id,)
                )
                if cursor.fetchone():
                    logger.warning("Cannot remove the 'Watch Later' playlist")
                    return False
                cursor.execute("DELETE FROM playlists WHERE id = ?", (playlist_id,))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Error removing playlist: {e}")
            return False

    @staticmethod
    def reorder_playlist_videos(playlist_id: int, video_ids: List[str]) -> bool:
        """
        Reorder videos in a playlist.
        Args:
            playlist_id: The ID of the playlist.
            video_ids: A list of video IDs in the new order.
        Returns:
            bool: True if the videos were reordered successfully, False otherwise.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("""
                    SELECT COUNT(*)
                    FROM playlist_videos
                    WHERE playlist_id = ? AND video_id IN ({})
                """.format(','.join(['?']*len(video_ids))), [playlist_id] + video_ids)
                if cursor.fetchone()[0] != len(video_ids):
                    logger.warning("Not all video IDs exist in the playlist")
                    return False
                for position, video_id in enumerate(video_ids):
                    cursor.execute("""
                        UPDATE playlist_videos
                        SET position = ?
                        WHERE playlist_id = ? AND video_id = ?
                    """, (position, playlist_id, video_id))
                cursor.execute("""
                    UPDATE playlists
                    SET updated_at = CURRENT_TIMESTAMP
                    WHERE id = ?
                """, (playlist_id,))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Error reordering playlist videos: {e}")
            return False

    @staticmethod
    def get_playlist_videos(playlist_id: int) -> List[Video]:
        """
        Get all videos in a specific playlist.
        Args:
            playlist_id: The ID of the playlist.
        Returns:
            List[Video]: A list of videos in the playlist.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute("""
                    SELECT video_id, title, url, thumbnail_url, duration,
                           platform, author, published
                    FROM playlist_videos
                    WHERE playlist_id = ?
                    ORDER BY position ASC
                """, (playlist_id,))
                videos = []
                for row in cursor.fetchall():
                    try:
                        video = Video(
                            title=row[1] or "No Title",
                            url=row[2] or "",
                            thumbnail_url=row[3] or "",
                            duration=row[4] or "0",
                            video_id=row[0],
                            platform=row[5] or "PeerTube",
                            author=row[6] or "Unknown",
                            published=datetime.fromisoformat(row[7]) if row[7] else None
                        )
                        videos.append(video)
                    except Exception as e:
                        logger.error(f"Error parsing video data: {e}")
                return videos
        except Exception as e:
            logger.error(f"Error retrieving playlist videos: {e}")
            return []

    @staticmethod
    def clear_playlist(playlist_id: int) -> bool:
        """
        Clear all videos from a playlist.
        Args:
            playlist_id: The ID of the playlist to clear.
        Returns:
            bool: True if the playlist was cleared successfully, False otherwise.
        """
        try:
            with db_manager.get_connection() as conn:
                cursor = conn.cursor()
                cursor.execute(
                    "SELECT 1 FROM playlists WHERE id = ? AND name = 'Watch Later'",
                    (playlist_id,)
                )
                if cursor.fetchone():
                    logger.warning("Cannot clear the 'Watch Later' playlist")
                    return False
                cursor.execute("DELETE FROM playlist_videos WHERE playlist_id = ?", (playlist_id,))
                conn.commit()
                return True
        except Exception as e:
            logger.error(f"Error clearing playlist: {e}")
            return False

    @staticmethod
    def export_playlist_to_opml(playlist_id: int, file_path: str) -> bool:
        """
        Export a playlist to OPML format.
        Args:
            playlist_id: The ID of the playlist to export.
            file_path: The path to save the OPML file to.
        Returns:
            bool: True if the export was successful, False otherwise.
        """
        try:
            import xml.etree.ElementTree as ET
            from xml.dom import minidom
            playlist = next((p for p in PlaylistService.get_playlists() if p.id == playlist_id), None)
            if not playlist:
                logger.error(f"Playlist with ID {playlist_id} not found")
                return False
            root = ET.Element("opml", version="2.0")
            head = ET.SubElement(root, "head")
            ET.SubElement(head, "title").text = f"PurpleTube - {playlist.name} Export"
            ET.SubElement(head, "dateCreated").text = datetime.now().isoformat()
            body = ET.SubElement(root, "body")
            playlist_outline = ET.SubElement(body, "outline", text=playlist.name)
            for video in playlist.videos:
                ET.SubElement(
                    playlist_outline, "outline",
                    type="video",
                    text=video.title,
                    url=video.url,
                    videoId=video.video_id,
                    duration=video.duration,
                    author=video.author,
                    published=video.published.isoformat() if video.published else ""
                )
            xml_str = minidom.parseString(ET.tostring(root)).toprettyxml(indent="  ", encoding="utf-8")
            with open(file_path, "w", encoding="utf-8") as f:
                f.write(xml_str.decode("utf-8"))
            logger.info(f"Successfully exported playlist to {file_path}")
            return True
        except Exception as e:
            logger.error(f"Error exporting playlist to OPML: {e}")
            return False

    @staticmethod
    def import_playlist_from_opml(file_path: str) -> bool:
        """
        Import a playlist from OPML format.
        Args:
            file_path: The path to the OPML file to import.
        Returns:
            bool: True if the import was successful, False otherwise.
        """
        try:
            import xml.etree.ElementTree as ET
            with open(file_path, "r", encoding="utf-8") as f:
                xml_content = f.read()
            root = ET.fromstring(xml_content)
            playlist_outline = root.find(".//outline[@text]")
            if not playlist_outline:
                logger.error("No playlist found in OPML file")
                return False
            playlist_name = playlist_outline.get("text", "Imported Playlist")
            playlist = Playlist(name=playlist_name)
            playlist_id = PlaylistService.save_playlist(playlist)
            if playlist_id == -1:
                logger.error("Failed to create playlist")
                return False
            for video_outline in playlist_outline.findall("outline[@type='video']"):
                try:
                    video = Video(
                        title=video_outline.get("text", "No Title"),
                        url=video_outline.get("url", ""),
                        thumbnail_url="",
                        duration=video_outline.get("duration", "0"),
                        video_id=video_outline.get("videoId", ""),
                        platform="PeerTube",
                        author=video_outline.get("author", "Unknown"),
                        published=datetime.fromisoformat(video_outline.get("published", "")) if video_outline.get("published") else None
                    )
                    if not PlaylistService.add_to_playlist(playlist_id, video):
                        logger.warning(f"Failed to add video {video.title} to playlist")
                except Exception as e:
                    logger.error(f"Error importing video: {e}")
            logger.info(f"Successfully imported playlist from {file_path}")
            return True
        except Exception as e:
            logger.error(f"Error importing playlist from OPML: {e}")
            return False

