"""Business logic services"""

import requests
from typing import List, Tuple, Optional
from functools import lru_cache

from .models import Video
from config.settings import SEPIASEARCH_BASE_URL, DEFAULT_SEARCH_COUNT

class SearchService:
    """Handles video search operations"""
    
    def __init__(self):
        self.session = requests.Session()
        self.session.headers.update({"User-Agent": "purpleTubePlayer/1.0"})
    
    @lru_cache(maxsize=32)
    def search_videos(self, query: str, page: int = 1, count: int = None) -> Tuple[List[Video], int]:
        """Search for videos with caching"""
        if count is None:
            count = DEFAULT_SEARCH_COUNT
            
        try:
            params = {
                'search': query,
                'start': (page-1)*count,
                'count': count,
                'nsfw': 'false',
                'sort': '-match'
            }
            
            response = self.session.get(SEPIASEARCH_BASE_URL, params=params, timeout=10)
            response.raise_for_status()
            
            data = response.json()
            videos = []
            
            for item in data.get("data", []):
                video = Video(
                    title=item.get("name", "No Title"),
                    url=item.get("url", ""),
                    thumbnail_url=item.get("thumbnailUrl", ""),
                    duration=str(item.get("duration", 0)),
                    video_id=str(item.get("uuid", "")),
                    author=item.get("account", {}).get("name", "Unknown")
                )
                videos.append(video)
                
            return videos, data.get("total", 0)
            
        except Exception as e:
            print(f"Search error: {e}")
            return [], 0

class PlaylistService:
    """Handles playlist operations"""
    
    @staticmethod
    def get_playlists():
        """Get all playlists"""
        # Implementation would interact with database
        return []
    
    @staticmethod
    def save_playlist(playlist):
        """Save a playlist"""
        # Database interaction
        pass
