import sys
import os
import re
import vlc
import requests
import yt_dlp
import logging
import webbrowser
import random
import time
from pathlib import Path
from collections import deque
from urllib.parse import urlparse
from bs4 import BeautifulSoup
from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QFrame, QSlider, QPushButton,
    QListWidget, QLabel, QListWidgetItem, QApplication, QMessageBox,
    QSizePolicy, QMenu, QStyle, QFileDialog, QProgressDialog, QInputDialog, QCheckBox
)
from PyQt5.QtCore import (
    Qt, QThread, pyqtSignal, QTimer, QPoint, QRect, QThreadPool, QRunnable, QObject,
    QPropertyAnimation, QEasingCurve
)
from PyQt5.QtGui import QPixmap, QMovie, QIcon, QCursor, QFont

# --- Import Services ---
from core.services.playlist_service import PlaylistService
from core.services.rss_service import RSSService
from core.services.search_service import SearchService

# --- Logging ---
logger = logging.getLogger(__name__)

# --- Theme ---
class Theme:
    PRIMARY = "#6a5acd"
    PRIMARY_HOVER = "#7a6bdb"
    PRIMARY_PRESSED = "#5a4bc9"
    SECONDARY = "#AAAAAA"
    SECONDARY_HOVER = "#CCCCCC"
    TERTIARY = "#FF9800"
    TERTIARY_HOVER = "#E68A00"
    BACKGROUND = "#0F0F0F"
    BACKGROUND_LIGHT = "#181818"
    SURFACE = "#212121"
    SURFACE_HOVER = "#2A2A2A"
    SURFACE_PRESSED = "#323232"
    TEXT_PRIMARY = "#FFFFFF"
    TEXT_SECONDARY = "#AAAAAA"
    TEXT_DISABLED = "#6B6B6B"
    TEXT_LINK = "#6a5acd"
    BORDER = "#3A3A3A"
    BORDER_ACTIVE = "#6a5acd"
    BORDER_HOVER = "#6B6B6B"
    SUCCESS = "#00FF66"
    DANGER = "#FF3333"
    WARNING = "#FFCC00"
    INFO = "#FF9800"
    FONT_FAMILY = "'Inter', 'Segoe UI', 'Roboto', sans-serif"

    @classmethod
    def player_controls_style(cls):
        return f"""
        #playerControls {{
            background: {cls.SURFACE};
            border-top: 1px solid {cls.BORDER};
            padding: 8px;
        }}
        #playPauseButton {{
            background: transparent;
            border: none;
            color: {cls.TEXT_PRIMARY};
            padding: 4px;
            border-radius: 4px;
        }}
        #mediaButton {{
            background: transparent;
            border: none;
            color: {cls.TEXT_PRIMARY};
            padding: 4px;
            border-radius: 4px;
        }}
        #mediaButton:hover {{
            background: rgba(106, 90, 205, 0.1);
        }}
        #playPauseButton:hover {{
            background: rgba(106, 90, 205, 0.1);
        }}
        QLabel {{
            padding: 0px;
            margin: 0px;
        }}
        """

    @classmethod
    def list_widget_style(cls):
        return f"""
        QListWidget {{
            background: {cls.SURFACE};
            border: 1px solid {cls.BORDER};
            border-radius: 4px;
            color: {cls.TEXT_PRIMARY};
            font-family: {cls.FONT_FAMILY};
            font-size: 14px;
            padding: 4px;
        }}
        QListWidget::item {{
            padding: 8px;
            border-radius: 4px;
        }}
        QListWidget::item:selected {{
            background: {cls.PRIMARY};
            color: white;
        }}
        """

# --- Models ---
class Video:
    def __init__(self, video_id, title, url, duration=0, author=None, published=None, thumbnail_url=None, platform=None):
        self.video_id = video_id
        self.title = title
        self.url = url
        self.duration = duration
        self.author = author
        self.published = published
        self.thumbnail_url = thumbnail_url
        self.platform = platform

class Playlist:
    def __init__(self, id=None, name="Nuova Playlist", videos=None):
        self.id = id
        self.name = name
        self.videos = videos if videos else []

# --- ShuffleWorker ---
class ShuffleWorkerSignals(QObject):
    shuffle_completed = pyqtSignal(list)
    progress_updated = pyqtSignal(int)
    error_occurred = pyqtSignal(str)

class ShuffleWorker(QRunnable):
    def __init__(self, queue, current_video=None):
        super().__init__()
        self.queue = queue.copy()
        self.current_video = current_video
        self.signals = ShuffleWorkerSignals()
        self._is_cancelled = False

    def run(self):
        try:
            logger.info("Starting shuffle operation...")
            
            # Emit initial progress
            self.signals.progress_updated.emit(10)
            
            # If there's a current video, keep it at position 0
            shuffled_queue = []
            if self.current_video and self.queue:
                # Remove current video from queue if present
                remaining_queue = [v for v in self.queue if getattr(v, 'video_id', None) != getattr(self.current_video, 'video_id', None)]
                
                # Shuffle the remaining videos using Fisher-Yates algorithm
                random.shuffle(remaining_queue)
                
                # Add current video back at position 0
                shuffled_queue = [self.current_video] + remaining_queue
                self.signals.progress_updated.emit(70)
            else:
                # Shuffle the entire queue
                shuffled_queue = self.queue.copy()
                random.shuffle(shuffled_queue)
                self.signals.progress_updated.emit(50)
            
            # Small delay to show progress (non-blocking)
            time.sleep(0.05)
            self.signals.progress_updated.emit(100)
            
            if not self._is_cancelled:
                logger.info(f"Shuffle completed. New queue length: {len(shuffled_queue)}")
                self.signals.shuffle_completed.emit(shuffled_queue)
            else:
                logger.info("Shuffle operation cancelled")
                
        except Exception as e:
            logger.error(f"Error during shuffle: {e}")
            self.signals.error_occurred.emit(str(e))

    def cancel(self):
        self._is_cancelled = True

# --- QueueManager ---
class QueueManager(QObject):
    queue_updated = pyqtSignal(list)
    next_video_started = pyqtSignal(object)
    shuffle_state_changed = pyqtSignal(bool)

    def __init__(self):
        super().__init__()
        self.queue = deque()
        self.current_video = None
        self.repeat = False
        self.is_shuffled = False
        self.original_queue_order = []
        self.shuffle_worker = None
        self.thread_pool = QThreadPool.globalInstance()

    def add_video(self, video):
        self.queue.append(video)
        logger.info(f"Added to queue: {getattr(video, 'title', 'Unknown')}")
        self.queue_updated.emit(list(self.queue))

    def remove_video(self, index):
        if 0 <= index < len(self.queue):
            del self.queue[index]
            self.queue_updated.emit(list(self.queue))

    def next_video(self):
        if not self.queue:
            return None
        if self.repeat:
            next_video = self.queue[0]
        else:
            next_video = self.queue.popleft()
        self.current_video = next_video
        self.next_video_started.emit(next_video)
        logger.info(f"Playing next video: {getattr(next_video, 'title', 'Unknown')}")
        return next_video

    def clear_queue(self):
        self.queue.clear()
        self.original_queue_order.clear()
        self.is_shuffled = False
        self.shuffle_state_changed.emit(False)
        self.queue_updated.emit(list(self.queue))

    def get_queue(self):
        return list(self.queue)

    def get_current_video(self):
        return self.current_video

    def toggle_repeat(self):
        self.repeat = not self.repeat
        return self.repeat

    def shuffle_queue(self):
        """Shuffle the queue using background thread"""
        if not self.queue:
            logger.info("Queue is empty, nothing to shuffle")
            return

        logger.info("Starting queue shuffle...")
        
        # Cancel any existing shuffle operation
        if self.shuffle_worker:
            self.shuffle_worker.cancel()

        # Store original order if this is the first shuffle
        if not self.is_shuffled and not self.original_queue_order:
            self.original_queue_order = list(self.queue)

        # Create and start shuffle worker
        self.shuffle_worker = ShuffleWorker(list(self.queue), self.current_video)
        self.shuffle_worker.signals.shuffle_completed.connect(self._on_shuffle_completed)
        self.shuffle_worker.signals.progress_updated.connect(self._on_shuffle_progress)
        self.shuffle_worker.signals.error_occurred.connect(self._on_shuffle_error)
        
        self.thread_pool.start(self.shuffle_worker)

    def _on_shuffle_completed(self, shuffled_queue):
        """Handle completed shuffle operation"""
        logger.info("Shuffle completed, updating queue...")
        
        # Update the queue
        self.queue.clear()
        self.queue.extend(shuffled_queue)
        
        self.is_shuffled = True
        self.shuffle_state_changed.emit(True)
        self.queue_updated.emit(list(self.queue))
        
        logger.info(f"Queue shuffled successfully. {len(self.queue)} items in queue.")

    def _on_shuffle_progress(self, progress):
        """Handle shuffle progress updates"""
        logger.debug(f"Shuffle progress: {progress}%")

    def _on_shuffle_error(self, error_msg):
        """Handle shuffle errors"""
        logger.error(f"Shuffle error: {error_msg}")
        # Use QTimer to safely show message box from non-main thread
        QTimer.singleShot(0, lambda: QMessageBox.warning(None, "Shuffle Error", f"Error during shuffle: {error_msg}"))

    def unshuffle_queue(self):
        """Restore original queue order"""
        if self.is_shuffled and self.original_queue_order:
            logger.info("Restoring original queue order...")
            self.queue.clear()
            self.queue.extend(self.original_queue_order)
            self.is_shuffled = False
            self.shuffle_state_changed.emit(False)
            self.queue_updated.emit(list(self.queue))
            logger.info("Original queue order restored")
        else:
            logger.info("Queue is not shuffled or no original order stored")

    def toggle_shuffle(self):
        """Toggle between shuffled and original order"""
        if self.is_shuffled:
            self.unshuffle_queue()
            return False
        else:
            self.shuffle_queue()
            return True

# --- CustomSlider ---
class CustomSlider(QSlider):
    def __init__(self, orientation):
        super().__init__(orientation)
        self._is_seeking = False
        self.setStyleSheet("""
            QSlider::groove:horizontal {
                border: none;
                height: 8px;
                background: #3a3a3a;
                border-radius: 4px;
            }
            QSlider::handle:horizontal {
                background: #BB86FC;
                border: none;
                width: 20px;
                height: 20px;
                margin: -6px 0;
                border-radius: 10px;
            }
            QSlider::handle:horizontal:hover {
                background: #9C27B0;
            }
            QSlider::sub-page:horizontal {
                background: qlineargradient(x1:0, y1:0, x2:1, y2:0, stop:0 #BB86FC, stop:1 #9C27B0);
                border-radius: 4px;
            }
        """)

    def mousePressEvent(self, event):
        self._is_seeking = True
        super().mousePressEvent(event)

    def mouseReleaseEvent(self, event):
        self._is_seeking = False
        super().mouseReleaseEvent(event)

# --- CustomVolumeSlider ---
class CustomVolumeSlider(QSlider):
    def __init__(self, orientation):
        super().__init__(orientation)
        self.setStyleSheet("""
            QSlider::groove:horizontal {
                height: 4px;
                background: #535353;
            }
            QSlider::sub-page:horizontal {
                background: #8E24AA;
            }
            QSlider::handle:horizontal {
                background: #8E24AA;
                width: 12px;
                height: 12px;
                margin: -4px 0;
            }
        """)
        self.setToolTip("Volume: 70%")
        self.valueChanged.connect(self.update_tooltip)

    def update_tooltip(self, value):
        self.setToolTip(f"Volume: {value}%")

# --- ControlBar ---
class ControlBar(QWidget):
    play_pause_clicked = pyqtSignal()
    next_clicked = pyqtSignal()
    previous_clicked = pyqtSignal()
    volume_changed = pyqtSignal(int)
    fullscreen_toggled = pyqtSignal()
    shuffle_clicked = pyqtSignal()

    def __init__(self):
        super().__init__()
        self.setObjectName("playerControls")
        self.setStyleSheet(Theme.player_controls_style())
        layout = QHBoxLayout(self)
        layout.setContentsMargins(12, 8, 12, 8)
        layout.setSpacing(8)

        # Volume state variables
        self.is_muted = False
        self.pre_mute_volume = 70

        # Play/Pause, Next, Previous buttons
        self.play_pause_btn = QPushButton()
        self.play_pause_btn.setObjectName("playPauseButton")
        self.play_pause_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaPlay))
        self.play_pause_btn.setFixedSize(36, 36)
        
        self.prev_btn = QPushButton()
        self.prev_btn.setObjectName("mediaButton")
        self.prev_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaSkipBackward))
        self.prev_btn.setFixedSize(36, 36)
        
        self.next_btn = QPushButton()
        self.next_btn.setObjectName("mediaButton")
        self.next_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaSkipForward))
        self.next_btn.setFixedSize(36, 36)

        # Shuffle button - using text icon since SP_MediaShuffle doesn't exist
        self.shuffle_btn = QPushButton("🔀")
        self.shuffle_btn.setObjectName("mediaButton")
        self.shuffle_btn.setFixedSize(36, 36)
        self.shuffle_btn.setToolTip("Attiva/disattiva riproduzione casuale")
        self.shuffle_btn.setCheckable(True)
        self.shuffle_btn.setStyleSheet(f"""
            QPushButton {{
                font-size: 16px;
                background: transparent;
                border: none;
                color: {Theme.TEXT_PRIMARY};
                padding: 4px;
                border-radius: 4px;
            }}
            QPushButton:hover {{
                background: rgba(106, 90, 205, 0.1);
            }}
            QPushButton:checked {{
                background: rgba(106, 90, 205, 0.3);
                border-radius: 4px;
            }}
        """)

        # Time label
        self.time_label = QLabel("00:00 / 00:00")
        self.time_label.setStyleSheet(f"""
            color: {Theme.TEXT_PRIMARY}; 
            font-family: {Theme.FONT_FAMILY};
            font-size: 14px;
            padding: 8px 0px;
        """)
        self.time_label.setFixedHeight(36)
        self.time_label.setAlignment(Qt.AlignCenter)

        # Volume button and slider container (always visible)
        volume_container = QWidget()
        volume_container.setFixedHeight(36)
        volume_layout = QHBoxLayout(volume_container)
        volume_layout.setContentsMargins(0, 0, 0, 0)
        volume_layout.setSpacing(4)
        volume_layout.setAlignment(Qt.AlignVCenter)

        self.volume_btn = QPushButton()
        self.volume_btn.setObjectName("mediaButton")
        self.volume_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaVolume))
        self.volume_btn.setFixedSize(36, 36)
        # Sposta l'icona del volume verso l'alto
        self.volume_btn.setStyleSheet("margin-top: -2px;")

        self.volume_slider = CustomVolumeSlider(Qt.Horizontal)
        self.volume_slider.setFixedWidth(175)
        self.volume_slider.setFixedHeight(36)
        self.volume_slider.setRange(0, 100)
        self.volume_slider.setValue(70)

        volume_layout.addWidget(self.volume_btn)
        volume_layout.addWidget(self.volume_slider)

        # Add widgets to the main layout
        layout.addWidget(self.shuffle_btn)
        layout.addWidget(self.prev_btn)
        layout.addWidget(self.play_pause_btn)
        layout.addWidget(self.next_btn)
        layout.addStretch()
        layout.addWidget(self.time_label)
        layout.addStretch()
        layout.addWidget(volume_container)

        # Allinea tutti i widget verticalmente al centro
        layout.setAlignment(Qt.AlignVCenter)

        # Connect signals
        self.play_pause_btn.clicked.connect(self.play_pause_clicked.emit)
        self.next_btn.clicked.connect(self.next_clicked.emit)
        self.prev_btn.clicked.connect(self.previous_clicked.emit)
        self.shuffle_btn.clicked.connect(self.shuffle_clicked.emit)
        self.volume_slider.valueChanged.connect(self.on_volume_changed)
        self.volume_btn.clicked.connect(self.toggle_mute)

    def toggle_mute(self):
        """Toggle mute state and update icon"""
        self.is_muted = not self.is_muted
        
        if self.is_muted:
            # Store current volume before muting
            self.pre_mute_volume = self.volume_slider.value()
            # Set volume to 0 and update icon
            self.volume_slider.setValue(0)
            self.volume_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaVolumeMuted))
        else:
            # Restore previous volume
            self.volume_slider.setValue(self.pre_mute_volume)
            self.update_volume_icon(self.pre_mute_volume)

    def on_volume_changed(self, value):
        """Handle volume changes and update icon"""
        if value > 0 and self.is_muted:
            self.is_muted = False
        self.update_volume_icon(value)
        self.volume_changed.emit(value)

    def update_volume_icon(self, volume):
        """Update volume icon based on volume level"""
        if volume == 0 or self.is_muted:
            self.volume_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaVolumeMuted))
            self.is_muted = True
        elif volume <= 33:
            self.volume_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaVolume))
        elif volume <= 66:
            self.volume_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaVolume))
        else:
            self.volume_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaVolume))

    def set_shuffle_state(self, is_shuffled):
        """Update shuffle button appearance"""
        self.shuffle_btn.setChecked(is_shuffled)
        if is_shuffled:
            self.shuffle_btn.setStyleSheet(f"""
                QPushButton {{
                    font-size: 16px;
                    background: rgba(106, 90, 205, 0.3);
                    border: none;
                    color: {Theme.TEXT_PRIMARY};
                    padding: 4px;
                    border-radius: 4px;
                }}
                QPushButton:hover {{
                    background: rgba(106, 90, 205, 0.4);
                }}
            """)
        else:
            self.shuffle_btn.setStyleSheet(f"""
                QPushButton {{
                    font-size: 16px;
                    background: transparent;
                    border: none;
                    color: {Theme.TEXT_PRIMARY};
                    padding: 4px;
                    border-radius: 4px;
                }}
                QPushButton:hover {{
                    background: rgba(106, 90, 205, 0.1);
                }}
            """)

# --- QueueWidget ---
class QueueWidget(QListWidget):
    video_selected = pyqtSignal(object)

    def __init__(self, queue_manager: QueueManager):
        super().__init__()
        self.queue_manager = queue_manager
        self.queue_manager.queue_updated.connect(self.update_queue_display)
        self.queue_manager.shuffle_state_changed.connect(self.on_shuffle_state_changed)
        self.setObjectName("queueWidget")
        self.setDragDropMode(QListWidget.InternalMove)
        self.setFixedWidth(300)
        self.setStyleSheet(Theme.list_widget_style())
        self.thumbnail_loaders = []
        self.itemDoubleClicked.connect(self.on_item_double_clicked)
        
        # Context menu
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self.show_context_menu)

    def on_item_double_clicked(self, item):
        row = self.row(item)
        queue = self.queue_manager.get_queue()
        if 0 <= row < len(queue):
            video = queue[row]
            self.video_selected.emit(video)

    def add_queue_item(self, video):
        item = QListWidgetItem()
        widget = QWidget()
        widget.setObjectName("queueItemWidget")
        widget_layout = QHBoxLayout(widget)
        widget_layout.setContentsMargins(8, 4, 8, 4)
        
        thumbnail = QLabel()
        thumbnail.setFixedSize(96, 54)
        thumbnail.setStyleSheet(f"background: {Theme.BORDER}; border-radius: 4px;")
        thumbnail.setPixmap(self.style().standardPixmap(QStyle.SP_FileIcon).scaled(96, 54, Qt.KeepAspectRatio))
        
        title_label = QLabel(video.title if hasattr(video, 'title') else "Video senza titolo")
        title_label.setObjectName("queueItemTitle")
        title_label.setStyleSheet(f"font-size: 14px; color: {Theme.TEXT_PRIMARY};")
        title_label.setWordWrap(True)
        
        widget_layout.addWidget(thumbnail)
        widget_layout.addWidget(title_label)
        
        item.setSizeHint(widget.sizeHint())
        self.addItem(item)
        self.setItemWidget(item, widget)

        if hasattr(video, 'thumbnail_url') and video.thumbnail_url:
            loader = ThumbnailLoader(video.thumbnail_url, thumbnail)
            loader.signals.thumbnail_loaded.connect(self.on_thumbnail_loaded)
            self.thumbnail_loaders.append(loader)
            QThreadPool.globalInstance().start(loader)

    def on_thumbnail_loaded(self, pixmap, label):
        if label and not label.isHidden():
            label.setPixmap(pixmap.scaled(96, 54, Qt.KeepAspectRatio))

    def update_queue_display(self):
        for loader in self.thumbnail_loaders:
            if hasattr(loader, 'stop'):
                loader.stop()
        self.thumbnail_loaders.clear()
        self.clear()
        logger.info(f"Updating queue display with {len(self.queue_manager.get_queue())} items")
        for video in self.queue_manager.get_queue():
            self.add_queue_item(video)

    def show_context_menu(self, position):
        menu = QMenu(self)
        menu.setStyleSheet(f"""
            QMenu {{
                background: {Theme.SURFACE};
                border: 1px solid {Theme.BORDER};
                border-radius: 6px;
                padding: 4px;
                color: {Theme.TEXT_PRIMARY};
            }}
            QMenu::item:selected {{
                background: rgba(106, 90, 205, 0.2);
                border-radius: 4px;
            }}
        """)
        
        item = self.itemAt(position)
        if item:
            remove_action = menu.addAction("Rimuovi dalla coda")
            menu.addSeparator()
        
        shuffle_action = menu.addAction("Mescola coda")
        unshuffle_action = menu.addAction("Ripristina ordine originale")
        menu.addSeparator()
        clear_action = menu.addAction("Svuota coda")
        
        action = menu.exec_(self.mapToGlobal(position))
        
        if item and action == remove_action:
            row = self.row(item)
            self.queue_manager.remove_video(row)
        elif action == shuffle_action:
            self.queue_manager.shuffle_queue()
        elif action == unshuffle_action:
            self.queue_manager.unshuffle_queue()
        elif action == clear_action:
            self.queue_manager.clear_queue()

    def on_shuffle_state_changed(self, is_shuffled):
        """Update display when shuffle state changes"""
        logger.info(f"Shuffle state changed: {is_shuffled}")
        # The queue display will be updated automatically via queue_updated signal

# --- ThumbnailLoader ---
class ThumbnailLoaderSignals(QObject):
    thumbnail_loaded = pyqtSignal(QPixmap, object)

class ThumbnailLoader(QRunnable):
    def __init__(self, url, label):
        super().__init__()
        self.signals = ThumbnailLoaderSignals()
        self.url = url
        self.label = label

    def run(self):
        try:
            response = requests.get(self.url, timeout=5)
            response.raise_for_status()
            pixmap = QPixmap()
            pixmap.loadFromData(response.content)
            self.signals.thumbnail_loaded.emit(pixmap, self.label)
        except Exception as e:
            logger.error(f"Error loading thumbnail: {e}")

    def stop(self):
        pass

# --- DirectURLWorker ---
class DirectURLWorkerSignals(QObject):
    finished = pyqtSignal(str)
    error = pyqtSignal(str)

class DirectURLWorker(QRunnable):
    def __init__(self, video_url: str):
        super().__init__()
        self.video_url = video_url
        self.signals = DirectURLWorkerSignals()

    def run(self):
        logger.info(f"Resolving direct URL for: {self.video_url}")
        try:
            ydl_opts = {
                'quiet': True,
                'no_warnings': True,
                'format': 'best[ext=mp4][protocol=https]',
                'get_url': True,
                'force_generic_extractor': True,
                'no_check_certificate': True,
                'user_agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                'playlist_items': '1',
                'no_playlist': True,
            }
            with yt_dlp.YoutubeDL(ydl_opts) as ydl:
                result = ydl.extract_info(self.video_url, download=False)
                if 'url' in result:
                    logger.info(f"Resolved direct URL: {result['url']}")
                    self.signals.finished.emit(result['url'])
                else:
                    logger.warning(f"No direct URL found for: {self.video_url}")
                    self.signals.error.emit("Nessun URL diretto trovato")
        except Exception as e:
            logger.error(f"Error resolving direct URL: {e}")
            self.signals.error.emit(str(e))

# --- DownloadWorker ---
class DownloadWorkerSignals(QObject):
    progress = pyqtSignal(int)
    finished = pyqtSignal(str, str)
    error = pyqtSignal(str, str)

class DownloadWorker(QRunnable):
    def __init__(self, url, download_dir):
        super().__init__()
        self.url = url
        self.download_dir = download_dir
        self._is_canceled = False
        self.signals = DownloadWorkerSignals()

    def stop(self):
        self._is_canceled = True
        logger.info("Download annullato dall'utente.")

    def run(self):
        ydl_opts = {
            'quiet': True,
            'no_warnings': True,
            'format': 'best',
            'outtmpl': f"{self.download_dir}/%(title)s.%(ext)s",
            'progress_hooks': [self._progress_hook],
        }
        try:
            with yt_dlp.YoutubeDL(ydl_opts) as ydl:
                ydl.download([self.url])
                if self._is_canceled:
                    self.signals.error.emit(self.url, "Download annullato dall'utente.")
                    return
                info = ydl.extract_info(self.url, download=False)
                filename = ydl.prepare_filename(info)
                self.signals.finished.emit(filename, "Download completato con successo!")
        except Exception as e:
            self.signals.error.emit(self.url, str(e))

    def _progress_hook(self, d):
        if self._is_canceled:
            raise Exception("Download annullato dall'utente.")
        if d['status'] == 'downloading':
            percent_str = d.get('_percent_str', '0%')
            percent_str = re.sub(r'\x1b\[[0-9;]*m', '', percent_str)
            try:
                progress = int(float(percent_str.strip('%')))
                self.signals.progress.emit(progress)
            except (ValueError, AttributeError):
                downloaded = d.get('downloaded_bytes', 0)
                total = d.get('total_bytes', 1)
                if total > 0:
                    progress = int((downloaded / total) * 100)
                    self.signals.progress.emit(progress)

# --- LoadingSpinner ---
class LoadingSpinner(QLabel):
    def __init__(self):
        super().__init__()
        self.movie = QMovie(":/icons/loading.gif")
        self.setMovie(self.movie)
        self.movie.start()
        self.setAlignment(Qt.AlignCenter)
        self.setStyleSheet("background: transparent;")

# --- ModernVLCPlayer ---
class ModernVLCPlayer(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setObjectName("playerPanel")
        self.setStyleSheet(Theme.player_controls_style())
        
        # Animation
        self.animation = QPropertyAnimation(self, b"windowOpacity")
        self.animation.setDuration(300)
        self.animation.setEasingCurve(QEasingCurve.InOutQuad)
        
        # VLC setup - Use simpler initialization first
        try:
            self.vlc_instance = vlc.Instance(self.get_optimized_vlc_args())
            if not self.vlc_instance:
                logger.warning("VLC instance creation failed with complex args, trying simple initialization")
                self.vlc_instance = vlc.Instance()
        except Exception as e:
            logger.error(f"VLC instance creation failed: {e}")
            try:
                self.vlc_instance = vlc.Instance()
            except Exception as e2:
                logger.error(f"VLC simple initialization also failed: {e2}")
                raise RuntimeError("VLC initialization failed completely")
        
        if not self.vlc_instance:
            raise RuntimeError("VLC instance is None - VLC may not be installed properly")
            
        self.vlc_player = self.vlc_instance.media_player_new()
        self.queue_manager = QueueManager()
        self.control_bar = ControlBar()
        self.custom_slider = CustomSlider(Qt.Horizontal)
        self.custom_slider.setObjectName("seekSlider")
        self.custom_slider.setFixedHeight(10)
        self.queue_widget = QueueWidget(self.queue_manager)
        self.queue_widget.video_selected.connect(self.on_video_selected)
        self.loading_spinner = LoadingSpinner()
        self.video_frame = QFrame(self)
        self.video_frame.setObjectName("videoFrame")
        self.video_frame.setMinimumSize(640, 360)
        self.video_frame.setContextMenuPolicy(Qt.CustomContextMenu)
        self.video_frame.customContextMenuRequested.connect(self.show_video_menu)
        self.playlist_service = PlaylistService()
        self.rss_service = RSSService()
        self.search_service = SearchService()

        # Layout setup
        main_layout = QHBoxLayout(self)
        main_layout.setContentsMargins(0, 0, 0, 0)
        main_layout.setSpacing(0)
        video_controls_widget = QWidget()
        video_controls_layout = QVBoxLayout(video_controls_widget)
        video_controls_layout.setContentsMargins(0, 0, 0, 0)
        video_controls_layout.setSpacing(0)
        video_controls_layout.addWidget(self.video_frame, stretch=1)
        video_controls_layout.addWidget(self.custom_slider)
        video_controls_layout.addWidget(self.control_bar)
        main_layout.addWidget(video_controls_widget, stretch=1)
        main_layout.addWidget(self.queue_widget)

        # VLC setup
        self.video_frame.show()
        win_id = self.video_frame.winId()
        self.attach_vlc_to_frame(win_id)

        # Connect signals
        self.queue_manager.next_video_started.connect(self.play)
        self.queue_manager.shuffle_state_changed.connect(self.control_bar.set_shuffle_state)
        self.control_bar.play_pause_clicked.connect(self.toggle_play_pause)
        self.control_bar.next_clicked.connect(self.play_next)
        self.control_bar.previous_clicked.connect(self.play_previous)
        self.control_bar.shuffle_clicked.connect(self.toggle_shuffle)
        self.control_bar.volume_changed.connect(self.set_volume)
        self.custom_slider.sliderMoved.connect(self.seek)
        self.custom_slider.sliderReleased.connect(self.on_seek_release)

        # VLC event manager
        self.event_manager = self.vlc_player.event_manager()
        self.event_manager.event_attach(vlc.EventType.MediaPlayerPlaying, self.on_playing)
        self.event_manager.event_attach(vlc.EventType.MediaPlayerPaused, self.on_paused)
        self.event_manager.event_attach(vlc.EventType.MediaPlayerStopped, self.on_stopped)
        self.event_manager.event_attach(vlc.EventType.MediaPlayerEndReached, self.on_end_reached)
        self.event_manager.event_attach(vlc.EventType.MediaPlayerEncounteredError, self.on_error)
        self.event_manager.event_attach(vlc.EventType.MediaPlayerMediaChanged, self.on_media_changed)
        self.event_manager.event_attach(vlc.EventType.MediaPlayerLengthChanged, self.on_length_changed)

        # Timers
        self.timer = QTimer(self)
        self.timer.timeout.connect(self.update_ui)
        self.timer.start(200)
        
        self.thread_pool = QThreadPool.globalInstance()
        self.seek_worker_pool = QThreadPool()
        self.seek_worker_pool.setMaxThreadCount(2)
        
        self.duration_poll_timer = QTimer(self)
        self.duration_poll_timer.timeout.connect(self.poll_duration)

        # Cursor timer for fullscreen
        self.cursor_timer = QTimer(self)
        self.cursor_timer.timeout.connect(self.hide_cursor)
        self.cursor_timer.setSingleShot(True)

    # --- Shuffle Methods ---
    def toggle_shuffle(self):
        """Toggle shuffle mode"""
        is_shuffled = self.queue_manager.toggle_shuffle()
        logger.info(f"Shuffle mode: {'enabled' if is_shuffled else 'disabled'}")
        
    def shuffle_queue(self):
        """Shuffle the current queue"""
        self.queue_manager.shuffle_queue()
        
    def unshuffle_queue(self):
        """Restore original queue order"""
        self.queue_manager.unshuffle_queue()

    # --- Cursor Management Methods ---
    def show_cursor(self):
        self.setCursor(QCursor(Qt.ArrowCursor))

    def hide_cursor(self):
        self.setCursor(QCursor(Qt.BlankCursor))

    def enterEvent(self, event):
        if self.isFullScreen():
            self.show_cursor()
            self.cursor_timer.start(3000)

    def leaveEvent(self, event):
        self.cursor_timer.stop()

    def mouseMoveEvent(self, event):
        if self.isFullScreen():
            self.show_cursor()
            self.cursor_timer.start(3000)
        super().mouseMoveEvent(event)

    # --- Context Menu ---
    def show_video_menu(self, pos):
        menu = QMenu(self)
        menu.setStyleSheet(f"""
            QMenu {{
                background: {Theme.SURFACE};
                border: 1px solid {Theme.BORDER};
                border-radius: 6px;
                padding: 4px;
                color: {Theme.TEXT_PRIMARY};
            }}
            QMenu::item:selected {{
                background: rgba(106, 90, 205, 0.2);
                border-radius: 4px;
            }}
        """)
        current_video = self.queue_manager.get_current_video()
        if current_video and hasattr(current_video, 'url'):
            menu.addAction("Apri con il browser", lambda: webbrowser.open(current_video.url))
            menu.addAction("Copia link", lambda: QApplication.clipboard().setText(current_video.url))
            menu.addAction("Scarica video", lambda: self.download_current_video())
            menu.addAction("Condividi su Mastodon", lambda: self.share_on_mastodon(current_video))
            menu.addAction("Aggiungi alla coda", lambda: self.queue_manager.add_video(current_video))
            menu.addSeparator()
            
            # Shuffle options
            shuffle_menu = menu.addMenu("Opzioni riproduzione casuale")
            shuffle_menu.addAction("Mescola coda", self.shuffle_queue)
            shuffle_menu.addAction("Ripristina ordine originale", self.unshuffle_queue)
            menu.addSeparator()

            # --- Playlist Submenu ---
            playlist_menu = menu.addMenu("Aggiungi alla playlist")
            playlists = self.playlist_service.get_playlists()
            for pl in playlists:
                action = playlist_menu.addAction(pl.name)
                action.triggered.connect(lambda _, p=pl.id: self.add_to_playlist(p, current_video))
            playlist_menu.addAction("Crea nuova playlist", self.create_new_playlist)

            # --- RSS Feed Option (PeerTube only) ---
            if hasattr(current_video, 'platform') and current_video.platform == "PeerTube":
                menu.addAction("Aggiungi a RSS Feed", lambda: self.add_to_rss_feed(current_video))
        else:
            menu.addAction("Nessun video in riproduzione", lambda: None)
        menu.exec_(self.video_frame.mapToGlobal(pos))

    # --- Playlist Management ---
    def add_to_playlist(self, playlist_id, video):
        if self.playlist_service.add_to_playlist(playlist_id, video):
            playlist_name = next((p.name for p in self.playlist_service.get_playlists() if p.id == playlist_id), "Sconosciuta")
            QMessageBox.information(self, "Successo", f"Video aggiunto a {playlist_name}")
        else:
            QMessageBox.warning(self, "Attenzione", "Il video è già presente in questa playlist")

    def create_new_playlist(self):
        name, ok = QInputDialog.getText(self, "Nuova Playlist", "Inserisci il nome della playlist:")
        if ok and name:
            pl = Playlist(name=name)
            playlist_id = self.playlist_service.save_playlist(pl)
            current_video = self.queue_manager.get_current_video()
            if current_video and self.playlist_service.add_to_playlist(playlist_id, current_video):
                QMessageBox.information(self, "Successo", f"Playlist '{name}' creata e video aggiunto!")

    # --- RSS Feed Management ---
    def add_to_rss_feed(self, video):
        """Aggiungi il canale del video a un feed RSS."""
        logger.info(f"Tentativo di aggiungere il feed RSS per il video: {video.title}")
        if not hasattr(video, 'url') or not video.url:
            logger.error("URL del video non disponibile.")
            QMessageBox.warning(self, "Errore", "URL del video non disponibile.")
            return

        try:
            logger.info(f"Estrazione dell'ID del canale dall'URL del video: {video.url}")
            channel_id = self.extract_channel_id(video.url)
            if not channel_id:
                logger.error("Impossibile estrarre l'ID del canale.")
                QMessageBox.warning(self, "Errore", "Impossibile estrarre l'ID del canale dal video.")
                return

            feed_url = self.generate_rss_url(video.url, channel_id)
            logger.info(f"URL del feed RSS generato: {feed_url}")
            existing_feeds = self.rss_service.get_feeds()
            existing_urls = [feed.url for feed in existing_feeds]
            logger.info(f"URL dei feed esistenti: {existing_urls}")

            if feed_url in existing_urls:
                logger.warning("Il feed RSS esiste già.")
                QMessageBox.warning(self, "Attenzione", "Questo feed RSS è già stato aggiunto.")
                return

            logger.info("Aggiunta del feed RSS al database...")
            if self.rss_service.add_feed(feed_url):
                logger.info("Feed RSS aggiunto con successo!")
                QMessageBox.information(
                    self,
                    "Successo",
                    f"Feed RSS del canale '{video.author}' aggiunto con successo!"
                )
                if hasattr(self, 'main_window') and self.main_window and hasattr(self.main_window, 'statusBar'):
                    QTimer.singleShot(0, lambda: self.main_window.statusBar().showMessage(
                        f"✓ Feed RSS del canale '{video.author}' aggiunto."
                    ))
            else:
                logger.error("Impossibile aggiungere il feed RSS.")
                QMessageBox.warning(self, "Errore", "Impossibile aggiungere il feed RSS del canale.")
        except Exception as e:
            logger.error(f"Errore durante l'aggiunta del feed RSS: {e}")
            QMessageBox.warning(
                self,
                "Errore",
                f"Errore durante l'aggiunta del feed RSS: {str(e)}"
            )

    def extract_channel_id(self, video_url: str) -> str:
        """
        Estrai l'ID del canale da un URL video PeerTube.
        Prova diversi metodi per trovare l'ID del canale.
        """
        try:
            logger.info(f"Estrazione dell'ID del canale dall'URL: {video_url}")
            response = requests.get(video_url, timeout=10)
            soup = BeautifulSoup(response.text, 'html.parser')

            # Metodo 1: Cerca il link RSS feed nell'HTML
            rss_link = soup.find('link', {'type': 'application/rss+xml'})
            if rss_link and 'href' in rss_link.attrs:
                href = rss_link['href']
                if 'videoChannelId=' in href:
                    return href.split('videoChannelId=')[1]

            # Metodo 2: Estrai dai dati JSON-LD
            script = soup.find('script', type='application/ld+json')
            if script:
                import json
                try:
                    data = json.loads(script.string)
                    if isinstance(data, list):
                        for item in data:
                            if item.get('@type') == 'VideoObject' and 'author' in item:
                                author_url = item['author'].get('url')
                                if author_url and 'video-channels' in author_url:
                                    return author_url.split('/')[-1]
                    elif isinstance(data, dict) and data.get('@type') == 'VideoObject' and 'author' in data:
                        author_url = data['author'].get('url')
                        if author_url and 'video-channels' in author_url:
                            return author_url.split('/')[-1]
                except Exception as e:
                    logger.error(f"Errore nel parsing JSON-LD: {e}")

            # Metodo 3: Estrai dal percorso URL del video (se è un video del canale)
            parts = urlparse(video_url).path.split('/')
            if 'c' in parts:
                channel_index = parts.index('c') + 1
                if channel_index < len(parts):
                    return parts[channel_index]

            # Metodo 4: Fallback API
            video_uuid = video_url.split('/')[-1]
            api_url = f"https://{urlparse(video_url).netloc}/api/v1/videos/{video_uuid}"
            api_response = requests.get(api_url, timeout=10)
            if api_response.status_code == 200:
                api_data = api_response.json()
                if 'channel' in api_data and 'id' in api_data['channel']:
                    return str(api_data['channel']['id'])
        except Exception as e:
            logger.error(f"Errore durante l'estrazione dell'ID del canale: {e}")
        return None

    def generate_rss_url(self, video_url: str, channel_id: str) -> str:
        """Genera l'URL del feed RSS per un canale PeerTube."""
        base_domain = f"{urlparse(video_url).scheme}://{urlparse(video_url).netloc}"
        return f"{base_domain}/feeds/videos.xml?videoChannelId={channel_id}"

    # --- Download ---
    def download_current_video(self):
        current_video = self.queue_manager.get_current_video()
        if current_video and hasattr(current_video, 'url'):
            download_dir = QFileDialog.getExistingDirectory(self, "Seleziona la cartella di destinazione", str(Path.home()))
            if download_dir:
                progress_dialog = QProgressDialog(
                    f"Download in corso: {current_video.title}",
                    "Annulla",
                    0,
                    100,
                    self
                )
                progress_dialog.setWindowTitle("Download in corso")
                progress_dialog.setWindowModality(Qt.WindowModal)
                progress_dialog.setAutoClose(False)
                progress_dialog.setValue(0)

                def update_progress(progress: int):
                    progress_dialog.setValue(progress)

                def on_completed(filename: str, message: str):
                    if not progress_dialog.wasCanceled():
                        progress_dialog.setValue(100)
                        QMessageBox.information(self, "Successo", f"Video salvato: {filename}")
                    progress_dialog.close()

                def on_error(url: str, error: str):
                    if not progress_dialog.wasCanceled():
                        QMessageBox.warning(self, "Errore", f"Download fallito: {error}")
                    progress_dialog.close()

                worker = DownloadWorker(current_video.url, download_dir)
                worker.signals.progress.connect(update_progress)
                worker.signals.finished.connect(on_completed)
                worker.signals.error.connect(on_error)
                progress_dialog.canceled.connect(worker.stop)

                self.thread_pool.start(worker)
                progress_dialog.exec_()

    # --- Playback ---
    def play(self, video=None, url=None):
        if video:
            self.queue_manager.current_video = video
        if url:
            self._play_direct_url(url, video)
        elif video is not None and hasattr(video, 'url'):
            worker = DirectURLWorker(video.url)
            worker.signals.finished.connect(lambda direct_url: self._play_direct_url(direct_url, video))
            worker.signals.error.connect(lambda msg: QMessageBox.warning(self, "Errore", f"Impossibile risolvere l'URL: {msg}"))
            QThreadPool.globalInstance().start(worker)

    def _play_direct_url(self, direct_url, video=None):
        if video:
            self.queue_manager.current_video = video
        if not direct_url:
            return
        media = self.vlc_instance.media_new(direct_url)
        
        # Enhanced audio and network options for better audio playback
        media.add_option(":network-caching=3000")
        media.add_option(":file-caching=3000")
        media.add_option(":audio-caching=3000")
        media.add_option(":sout-keep")
        media.add_option(":audio-track=0")
        media.add_option(":no-audio-desync")
        media.add_option(":input-fast-seek")
        media.add_option(":http-reconnect")
        media.add_option(f":http-user-agent=Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36")
        if video and hasattr(video, 'duration'):
            duration_sec = int(video.duration)
            self.custom_slider.setRange(0, duration_sec)
            self.control_bar.time_label.setText(f"00:00 / {self.format_time(duration_sec)}")
        media.parse_with_options(vlc.MediaParseFlag.local, -1)
        self.vlc_player.set_media(media)
        
        # Ensure audio is enabled and volume set
        self.vlc_player.audio_set_volume(70)
        self.vlc_player.audio_set_mute(False)
        
        # Reset seekbar to start when new video begins playing
        self.custom_slider.setValue(0)
        self.control_bar.time_label.setText("00:00 / 00:00")
        
        self.video_frame.update()
        self.video_frame.updateGeometry()
        self.vlc_player.play()
        
        # Force audio track selection after play starts
        QTimer.singleShot(100, self.ensure_audio_track)
        
        self.duration_poll_timer.start(200)

    def ensure_audio_track(self):
        """Ensure an audio track is selected"""
        try:
            audio_tracks = self.vlc_player.audio_get_track_description()
            if audio_tracks:
                # Select first available audio track
                self.vlc_player.audio_set_track(0)
                logger.info("Audio track set to 0")
        except Exception as e:
            logger.error(f"Error setting audio track: {e}")

    # --- VLC Helpers ---
    def attach_vlc_to_frame(self, win_id):
        if sys.platform.startswith('linux'):
            self.vlc_player.set_xwindow(int(win_id))
        elif sys.platform == 'win32':
            self.vlc_player.set_hwnd(int(win_id))
        elif sys.platform == 'darwin':
            self.vlc_player.set_nsobject(int(win_id))

    def get_optimized_vlc_args(self):
        """Get optimized VLC arguments with fallback to simple args if complex ones fail"""
        try:
            # Try simpler, more reliable arguments first
            args = [
                "--network-caching=3000",
                "--file-caching=3000", 
                "--live-caching=3000",
                "--avcodec-threads=0",
                "--no-xlib",  # Important for some Linux systems
            ]
            
            # Platform-specific audio output
            if sys.platform == "win32":
                args.append("--aout=directsound")
            elif sys.platform == "linux":
                args.append("--aout=pulse")
            elif sys.platform == "darwin":
                args.append("--aout=coreaudio")
                
            return args
        except Exception as e:
            logger.warning(f"Error generating VLC args, using minimal set: {e}")
            return []  # Return empty list for default initialization

    # --- UI Updates ---
    def update_ui(self):
        if self.vlc_player.get_media():
            try:
                duration_ms = self.vlc_player.get_media().get_duration()
                current_time_ms = self.vlc_player.get_time()
                if duration_ms > 0 and current_time_ms >= 0:
                    duration_sec = duration_ms // 1000
                    current_time_sec = current_time_ms // 1000
                    if not hasattr(self.custom_slider, '_is_seeking') or not self.custom_slider._is_seeking:
                        self.custom_slider.setRange(0, duration_sec)
                        self.custom_slider.setValue(current_time_sec)
                        self.control_bar.time_label.setText(
                            f"{self.format_time(current_time_sec)} / {self.format_time(duration_sec)}"
                        )
            except Exception as e:
                logger.error(f"UI update error: {e}")

    def format_time(self, seconds):
        minutes, seconds = divmod(int(seconds), 60)
        hours, minutes = divmod(minutes, 60)
        if hours > 0:
            return f"{hours}:{minutes:02d}:{seconds:02d}"
        else:
            return f"{minutes:02d}:{seconds:02d}"

    # --- Event Handlers ---
    def on_video_selected(self, video):
        if hasattr(video, 'url'):
            self.play(video)

    def toggle_play_pause(self):
        if self.vlc_player.is_playing():
            self.vlc_player.pause()
            self.control_bar.play_pause_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaPlay))
        else:
            self.vlc_player.play()
            self.control_bar.play_pause_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaPause))

    def play_next(self):
        next_video = self.queue_manager.next_video()
        if next_video:
            self.play(next_video)

    def play_previous(self):
        pass

    def seek(self, position):
        if self.vlc_player.get_media() and self.vlc_player.is_playing():
            self.custom_slider._is_seeking = True
            position_ms = position * 1000
            self.vlc_player.set_time(position_ms)

    def on_seek_release(self):
        if self.vlc_player.get_media():
            self.custom_slider._is_seeking = False
            position = self.custom_slider.value() * 1000
            self.vlc_player.set_time(position)

    def set_volume(self, volume):
        self.vlc_player.audio_set_volume(volume)

    def on_playing(self, event):
        self.poll_duration()
        QTimer.singleShot(1000, self.check_audio_info)

    def on_paused(self, event):
        self.control_bar.play_pause_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaPlay))

    def on_stopped(self, event):
        self.control_bar.play_pause_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaPlay))

    def on_end_reached(self, event):
        self.play_next()

    def on_error(self, event):
        logger.error(f"VLC Error: {event}")
        self.control_bar.play_pause_btn.setIcon(self.style().standardIcon(QStyle.SP_MediaPlay))
        self.check_audio_info()

    def check_audio_info(self):
        """Debug audio information"""
        try:
            volume = self.vlc_player.audio_get_volume()
            is_muted = self.vlc_player.audio_get_mute()
            track_count = self.vlc_player.audio_get_track_count()
            current_track = self.vlc_player.audio_get_track()
            
            logger.info(f"Audio Info - Volume: {volume}, Muted: {is_muted}")
            logger.info(f"Audio Tracks - Count: {track_count}, Current: {current_track}")
            
            if track_count == 0:
                logger.warning("No audio tracks available in current media")
                
        except Exception as e:
            logger.error(f"Error checking audio info: {e}")

    def poll_duration(self):
        if self.vlc_player.get_media():
            duration_ms = self.vlc_player.get_media().get_duration()
            if duration_ms > 0:
                duration_sec = duration_ms // 1000
                self.custom_slider.setRange(0, duration_sec)
                self.control_bar.time_label.setText(f"{self.format_time(self.vlc_player.get_time() // 1000)} / {self.format_time(duration_sec)}")
                self.duration_poll_timer.stop()

    def on_length_changed(self, event):
        self.poll_duration()

    def on_media_changed(self, event):
        self.poll_duration()

    def resizeEvent(self, event):
        super().resizeEvent(event)
        if hasattr(self, 'video_frame') and self.video_frame:
            self.video_frame.updateGeometry()
            win_id = self.video_frame.winId()
            self.attach_vlc_to_frame(win_id)

    def mousePressEvent(self, event):
        super().mousePressEvent(event)

    def reattach_video_surface(self):
        if hasattr(self, 'vlc_player') and hasattr(self, 'video_frame'):
            win_id = self.video_frame.winId()
            self.attach_vlc_to_frame(win_id)

    def repaint_video(self):
        if hasattr(self, 'video_frame') and self.video_frame:
            self.video_frame.update()

    def closeEvent(self, event):
        for loader in self.queue_widget.thumbnail_loaders:
            if hasattr(loader, 'stop'):
                loader.stop()
        event.accept()

    # --- Shortcut Methods ---
    def seek_forward(self):
        """Avanza di 10 secondi."""
        if self.vlc_player and self.vlc_player.get_media():
            current_time = self.vlc_player.get_time()
            self.vlc_player.set_time(current_time + 10000)

    def seek_backward(self):
        """Indietreggia di 10 secondi."""
        if self.vlc_player and self.vlc_player.get_media():
            current_time = self.vlc_player.get_time()
            self.vlc_player.set_time(max(0, current_time - 10000))

    def increase_volume(self):
        """Aumenta il volume del 10%."""
        if self.vlc_player:
            current_volume = self.vlc_player.audio_get_volume()
            self.vlc_player.audio_set_volume(min(100, current_volume + 10))
            self.control_bar.volume_slider.setValue(min(100, current_volume + 10))

    def decrease_volume(self):
        """Diminuisci il volume del 10%."""
        if self.vlc_player:
            current_volume = self.vlc_player.audio_get_volume()
            self.vlc_player.audio_set_volume(max(0, current_volume - 10))
            self.control_bar.volume_slider.setValue(max(0, current_volume - 10))

    def clear_queue(self):
        """Svuota la coda di riproduzione."""
        self.queue_manager.clear_queue()

    def repeat_queue(self):
        """Attiva/disattiva la ripetizione della coda."""
        self.queue_manager.toggle_repeat()
        QMessageBox.information(self, "Ripeti", f"Modalità ripetizione {'attivata' if self.queue_manager.repeat else 'disattivata'}")

    # --- Mastodon Sharing (FIXED) ---
    def share_on_mastodon(self, video):
        """Condividi video su Mastodon."""
        if not video or not hasattr(video, 'url') or not hasattr(video, 'title'):
            QMessageBox.warning(self, "Errore", "Nessun video valido da condividere.")
            return
            
        instance, ok = QInputDialog.getText(
            self,
            "Condividi su Mastodon",
            "Inserisci l'URL della tua istanza Mastodon (es. mastodon.social):"
        )
        if ok and instance:
            instance = instance.strip()
            if not instance.startswith(("http://", "https://")):
                instance = f"https://{instance}"
            
            # Create share URL with video title and URL
            share_text = f"Guardando: {video.title} {video.url}"
            share_url = f"{instance}/share?text={share_text}"
            webbrowser.open(share_url)

# --- Expose classes ---
__all__ = [
    'QueueManager',
    'DownloadWorker',
    'DirectURLWorker',
    'ModernVLCPlayer',
    'ControlBar',
    'QueueWidget',
    'ThumbnailLoader',
    'LoadingSpinner',
    'PlaylistService',
    'RSSService',
    'SearchService',
    'Video',
    'Playlist',
    'ShuffleWorker'
]
