"""
Complete data models for PurpleTube Player application.
This module contains all data structures used throughout the application,
including Video, Playlist, RSSFeed, and FeedItem models with full support
for all metadata fields and serialization capabilities.
"""
from dataclasses import dataclass, field, asdict
from typing import List, Optional, Dict, Any, Union
from datetime import datetime
import json
from collections import deque

@dataclass
class Video:
    """
    Video model representing a video from PeerTube or other platforms.
    Contains all metadata needed for display, playback, and management.
    """
    title: str
    url: str
    thumbnail_url: str
    duration: str
    video_id: str
    platform: str = "PeerTube"
    author: str = "Unknown"
    published: Optional[datetime] = None
    description: str = ""
    views: int = 0
    likes: int = 0
    dislikes: int = 0
    comments: int = 0
    language: Optional[str] = None
    category: Optional[str] = None
    channel_id: Optional[str] = None
    channel_url: Optional[str] = None
    channel_name: Optional[str] = None
    tags: List[str] = field(default_factory=list)
    is_live: bool = False
    nsfw: bool = False
    license: Optional[str] = None
    privacy: Optional[str] = None
    direct_url: Optional[str] = None
    subtitles: Dict[str, str] = field(default_factory=dict)
    related_videos: List[str] = field(default_factory=list)
    source: Optional[str] = None
    resolution: Optional[str] = None
    fps: Optional[float] = None
    size: Optional[int] = None  # Size in bytes
    bitrate: Optional[int] = None  # Bitrate in kbps
    format: Optional[str] = None
    metadata: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        """
        Convert the Video object to a dictionary.
        Handles datetime serialization and optional fields.
        """
        result = asdict(self)

        # Handle datetime serialization
        if result['published']:
            result['published'] = result['published'].isoformat()

        # Remove None values for cleaner serialization
        return {k: v for k, v in result.items() if v is not None and v != [] and v != {}}

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'Video':
        """
        Create a Video object from a dictionary.
        Handles datetime deserialization and missing fields.
        """
        if 'published' in data and data['published']:
            data['published'] = datetime.fromisoformat(data['published'])

        # Set defaults for missing fields
        defaults = {
            'platform': "PeerTube",
            'author': "Unknown",
            'description': "",
            'views': 0,
            'likes': 0,
            'dislikes': 0,
            'comments': 0,
            'tags': [],
            'subtitles': {},
            'related_videos': [],
            'metadata': {}
        }

        # Merge defaults with provided data
        for key, default in defaults.items():
            if key not in data:
                data[key] = default

        return cls(**data)

    def to_json(self) -> str:
        """Convert the Video object to a JSON string."""
        return json.dumps(self.to_dict())

    @classmethod
    def from_json(cls, json_str: str) -> 'Video':
        """Create a Video object from a JSON string."""
        return cls.from_dict(json.loads(json_str))

    def __hash__(self) -> int:
        """Make Video objects hashable based on their video_id."""
        return hash(self.video_id)

    def __eq__(self, other: object) -> bool:
        """Compare Video objects based on their video_id."""
        if not isinstance(other, Video):
            return False
        return self.video_id == other.video_id

@dataclass
class Playlist:
    """
    Playlist model representing a collection of videos.
    Supports adding, removing, and managing videos in the playlist.
    """
    name: str
    videos: List[Video] = field(default_factory=list)
    id: Optional[int] = None
    created_at: Optional[datetime] = field(default_factory=datetime.now)
    updated_at: Optional[datetime] = field(default_factory=datetime.now)
    description: str = ""
    is_public: bool = False
    thumbnail_url: Optional[str] = None

    def add_video(self, video: Video) -> bool:
        """
        Add a video to the playlist.
        Returns True if added, False if already exists.
        """
        if video.video_id not in [v.video_id for v in self.videos]:
            self.videos.append(video)
            self.updated_at = datetime.now()
            return True
        return False

    def remove_video(self, video_id: str) -> bool:
        """
        Remove a video from the playlist by its ID.
        Returns True if removed, False if not found.
        """
        for i, video in enumerate(self.videos):
            if video.video_id == video_id:
                self.videos.pop(i)
                self.updated_at = datetime.now()
                return True
        return False

    def clear(self) -> None:
        """Clear all videos from the playlist."""
        self.videos.clear()
        self.updated_at = datetime.now()

    def to_dict(self) -> Dict[str, Any]:
        """
        Convert the Playlist object to a dictionary.
        Handles datetime serialization and video list conversion.
        """
        result = asdict(self)
        result['videos'] = [v.to_dict() for v in self.videos]

        # Handle datetime serialization
        if result['created_at']:
            result['created_at'] = result['created_at'].isoformat()
        if result['updated_at']:
            result['updated_at'] = result['updated_at'].isoformat()

        return result

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'Playlist':
        """
        Create a Playlist object from a dictionary.
        Handles datetime deserialization and video list conversion.
        """
        if 'created_at' in data and data['created_at']:
            data['created_at'] = datetime.fromisoformat(data['created_at'])
        if 'updated_at' in data and data['updated_at']:
            data['updated_at'] = datetime.fromisoformat(data['updated_at'])

        if 'videos' in data:
            data['videos'] = [Video.from_dict(v) for v in data['videos']]

        return cls(**data)

    def to_json(self) -> str:
        """Convert the Playlist object to a JSON string."""
        return json.dumps(self.to_dict())

    @classmethod
    def from_json(cls, json_str: str) -> 'Playlist':
        """Create a Playlist object from a JSON string."""
        return cls.from_dict(json.loads(json_str))

@dataclass
class RSSFeed:
    """
    RSS Feed model representing a video feed from a PeerTube channel or other sources.
    """
    url: str
    title: str
    id: Optional[int] = None
    last_updated: Optional[datetime] = field(default_factory=datetime.now)
    is_channel: bool = False
    description: str = ""
    language: Optional[str] = None
    thumbnail_url: Optional[str] = None
    video_count: int = 0
    subscribers: int = 0
    metadata: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        """Convert the RSSFeed object to a dictionary."""
        result = asdict(self)

        # Handle datetime serialization
        if result['last_updated']:
            result['last_updated'] = result['last_updated'].isoformat()

        return result

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'RSSFeed':
        """Create an RSSFeed object from a dictionary."""
        if 'last_updated' in data and data['last_updated']:
            data['last_updated'] = datetime.fromisoformat(data['last_updated'])

        return cls(**data)

    def to_json(self) -> str:
        """Convert the RSSFeed object to a JSON string."""
        return json.dumps(self.to_dict())

    @classmethod
    def from_json(cls, json_str: str) -> 'RSSFeed':
        """Create an RSSFeed object from a JSON string."""
        return cls.from_dict(json.loads(json_str))

@dataclass
class FeedItem:
    """
    Feed Item model representing a single item from an RSS feed.
    Similar to Video but specifically for feed items.
    """
    title: str
    url: str
    thumbnail_url: str
    duration: str
    video_id: str
    published: datetime
    author: str
    platform: str = "PeerTube"
    description: str = ""
    views: int = 0
    likes: int = 0
    comments: int = 0
    language: Optional[str] = None
    category: Optional[str] = None
    channel_id: Optional[str] = None
    channel_url: Optional[str] = None
    tags: List[str] = field(default_factory=list)
    is_live: bool = False
    nsfw: bool = False
    metadata: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        """Convert the FeedItem object to a dictionary."""
        result = asdict(self)

        # Handle datetime serialization
        if result['published']:
            result['published'] = result['published'].isoformat()

        return result

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'FeedItem':
        """Create a FeedItem object from a dictionary."""
        if 'published' in data and data['published']:
            data['published'] = datetime.fromisoformat(data['published'])

        return cls(**data)

    def to_json(self) -> str:
        """Convert the FeedItem object to a JSON string."""
        return json.dumps(self.to_dict())

    @classmethod
    def from_json(cls, json_str: str) -> 'FeedItem':
        """Create a FeedItem object from a JSON string."""
        return cls.from_dict(json.loads(json_str))

    def to_video(self) -> Video:
        """Convert this FeedItem to a Video object."""
        return Video(
            title=self.title,
            url=self.url,
            thumbnail_url=self.thumbnail_url,
            duration=self.duration,
            video_id=self.video_id,
            platform=self.platform,
            author=self.author,
            published=self.published,
            description=self.description,
            views=self.views,
            likes=self.likes,
            comments=self.comments,
            language=self.language,
            category=self.category,
            channel_id=self.channel_id,
            channel_url=self.channel_url,
            tags=self.tags,
            is_live=self.is_live,
            nsfw=self.nsfw
        )

@dataclass
class SearchResult:
    """
    Search Result model representing a page of search results.
    """
    query: str
    videos: List[Video]
    total_results: int
    page: int
    page_size: int
    sort_by: str = "relevance"
    language: Optional[str] = None
    category: Optional[str] = None
    duration: Optional[str] = None
    published_after: Optional[str] = None
    metadata: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        """Convert the SearchResult object to a dictionary."""
        return {
            'query': self.query,
            'videos': [v.to_dict() for v in self.videos],
            'total_results': self.total_results,
            'page': self.page,
            'page_size': self.page_size,
            'sort_by': self.sort_by,
            'language': self.language,
            'category': self.category,
            'duration': self.duration,
            'published_after': self.published_after,
            'metadata': self.metadata
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'SearchResult':
        """Create a SearchResult object from a dictionary."""
        data['videos'] = [Video.from_dict(v) for v in data['videos']]
        return cls(**data)

    def to_json(self) -> str:
        """Convert the SearchResult object to a JSON string."""
        return json.dumps(self.to_dict())

    @classmethod
    def from_json(cls, json_str: str) -> 'SearchResult':
        """Create a SearchResult object from a JSON string."""
        return cls.from_dict(json.loads(json_str))

@dataclass
class QueueItem:
    """
    Queue Item model representing a video in the playback queue.
    """
    video: Video
    added_at: datetime = field(default_factory=datetime.now)
    played: bool = False
    position: int = 0

    def to_dict(self) -> Dict[str, Any]:
        """Convert the QueueItem object to a dictionary."""
        return {
            'video': self.video.to_dict(),
            'added_at': self.added_at.isoformat(),
            'played': self.played,
            'position': self.position
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'QueueItem':
        """Create a QueueItem object from a dictionary."""
        data['video'] = Video.from_dict(data['video'])
        data['added_at'] = datetime.fromisoformat(data['added_at'])
        return cls(**data)

    def to_json(self) -> str:
        """Convert the QueueItem object to a JSON string."""
        return json.dumps(self.to_dict())

    @classmethod
    def from_json(cls, json_str: str) -> 'QueueItem':
        """Create a QueueItem object from a JSON string."""
        return cls.from_dict(json.loads(json_str))

@dataclass
class PlaybackState:
    """
    Playback State model representing the current state of video playback.
    """
    current_video: Optional[Video] = None
    current_time: float = 0.0
    duration: float = 0.0
    paused: bool = True
    volume: int = 70
    muted: bool = False
    playback_rate: float = 1.0
    fullscreen: bool = False
    queue: List[QueueItem] = field(default_factory=list)
    history: List[Video] = field(default_factory=list)
    metadata: Dict[str, Any] = field(default_factory=dict)

    def to_dict(self) -> Dict[str, Any]:
        """Convert the PlaybackState object to a dictionary."""
        return {
            'current_video': self.current_video.to_dict() if self.current_video else None,
            'current_time': self.current_time,
            'duration': self.duration,
            'paused': self.paused,
            'volume': self.volume,
            'muted': self.muted,
            'playback_rate': self.playback_rate,
            'fullscreen': self.fullscreen,
            'queue': [item.to_dict() for item in self.queue],
            'history': [video.to_dict() for video in self.history],
            'metadata': self.metadata
        }

    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'PlaybackState':
        """Create a PlaybackState object from a dictionary."""
        if data.get('current_video'):
            data['current_video'] = Video.from_dict(data['current_video'])

        data['queue'] = [QueueItem.from_dict(item) for item in data.get('queue', [])]
        data['history'] = [Video.from_dict(video) for video in data.get('history', [])]

        return cls(**data)

    def to_json(self) -> str:
        """Convert the PlaybackState object to a JSON string."""
        return json.dumps(self.to_dict())

    @classmethod
    def from_json(cls, json_str: str) -> 'PlaybackState':
        """Create a PlaybackState object from a JSON string."""
        return cls.from_dict(json.loads(json_str))

