from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QLabel, QListWidget, QListWidgetItem,
    QPushButton, QHBoxLayout, QMenu, QMessageBox, QInputDialog, QGridLayout, QScrollArea, QSplitter
)
from PyQt5.QtCore import Qt, pyqtSignal, QTimer, QThreadPool, QRunnable, QUrl, QMetaObject, Q_ARG, pyqtSlot
from PyQt5.QtGui import QDesktopServices
from PyQt5.QtWidgets import QApplication
from core.models import Video, RSSFeed
from core.services.rss_service import RSSService
from core.ui.components.video_card import VideoCard
from config.theme import Theme
import logging

logger = logging.getLogger(__name__)

class FeedManagerPanel(QWidget):
    """Pannello per la gestione dei feed RSS con integrazione VLC, supporto alla coda e funzionalità complete di tasto destro."""
    video_card_play_requested = pyqtSignal(Video)  # Segnale per riprodurre un video
    add_to_queue_requested = pyqtSignal(Video)    # Segnale per aggiungere alla coda
    feed_selected = pyqtSignal(int)               # Segnale per la selezione del feed
    feed_refreshed = pyqtSignal()                 # Segnale per il completamento dell'aggiornamento del feed
    refresh_complete = pyqtSignal()               # Segnale personalizzato per il completamento dell'aggiornamento
    refresh_error = pyqtSignal(str)               # Segnale personalizzato per l'errore di aggiornamento

    def __init__(self, main_window, rss_service=None):
        super().__init__()
        self.main_window = main_window
        self.rss_service = rss_service or RSSService()
        self.current_feed = None
        self.all_videos = []
        self.current_offset = 0
        self.is_loading = False
        self.sort_newest_first = True
        self.refresh_in_progress = False
        self.setup_ui()
        # Connetti i segnali personalizzati
        self.refresh_complete.connect(self._handle_refresh_complete)
        self.refresh_error.connect(self._handle_refresh_error)
        # Timer per l'aggiornamento automatico
        self.refresh_timer = QTimer()
        self.refresh_timer.timeout.connect(self.refresh_all_feeds)
        self.refresh_timer.start(300000)  # Aggiorna ogni 5 minuti
        # Aggiornamento iniziale
        QTimer.singleShot(100, self.refresh_feeds)

    def setup_ui(self):
        """Inizializza l'interfaccia utente con la gestione dei feed a sinistra e la griglia dei video a destra."""
        main_layout = QHBoxLayout(self)
        main_layout.setContentsMargins(0, 0, 0, 0)
        main_layout.setSpacing(0)
        
        # Sidebar sinistra per i feed
        self.sidebar_widget = QWidget()
        self.sidebar_widget.setFixedWidth(280)  # Larghezza fissa per la sidebar
        sidebar_layout = QVBoxLayout(self.sidebar_widget)
        sidebar_layout.setContentsMargins(16, 16, 16, 16)
        sidebar_layout.setSpacing(12)
        
        # Titolo
        title_label = QLabel("Gestore Feed RSS")
        title_label.setStyleSheet(f"color: {Theme.TEXT_PRIMARY}; font-size: 16px; font-weight: bold;")
        sidebar_layout.addWidget(title_label)
        
        # Controlli feed
        controls_layout = QVBoxLayout()
        controls_layout.setContentsMargins(0, 0, 0, 0)
        controls_layout.setSpacing(8)
        
        # Pulsante aggiungi nuovo feed
        self.new_feed_btn = QPushButton("Aggiungi Feed")
        self.new_feed_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.PRIMARY};
                color: white;
                border: none;
                border-radius: 4px;
                padding: 8px 12px;
                font-family: {Theme.FONT_FAMILY};
            }}
            QPushButton:hover {{ background: {Theme.PRIMARY_HOVER}; }}
        """)
        self.new_feed_btn.clicked.connect(self.add_new_feed)
        
        # Pulsante elimina feed
        self.delete_feed_btn = QPushButton("Elimina Feed")
        self.delete_feed_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.DANGER};
                color: white;
                border: none;
                border-radius: 4px;
                padding: 8px 12px;
                font-family: {Theme.FONT_FAMILY};
            }}
            QPushButton:hover {{ background: #ff5555; }}
        """)
        self.delete_feed_btn.clicked.connect(self.delete_selected_feed)
        
        # Pulsante aggiorna tutto
        self.refresh_all_btn = QPushButton("Aggiorna Tutto")
        self.refresh_all_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.TERTIARY};
                color: white;
                border: none;
                border-radius: 4px;
                padding: 8px 12px;
                font-family: {Theme.FONT_FAMILY};
            }}
            QPushButton:hover {{ background: {Theme.TERTIARY_HOVER}; }}
        """)
        self.refresh_all_btn.clicked.connect(self.refresh_all_feeds)
        
        # Pulsante riproduci tutto
        self.play_all_btn = QPushButton("Riproduci Tutto")
        self.play_all_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.SECONDARY};
                color: white;
                border: none;
                border-radius: 4px;
                padding: 8px 12px;
                font-family: {Theme.FONT_FAMILY};
            }}
            QPushButton:hover {{ background: {Theme.SECONDARY}; }}
        """)
        self.play_all_btn.clicked.connect(self.play_all_videos)
        
        controls_layout.addWidget(self.new_feed_btn)
        controls_layout.addWidget(self.delete_feed_btn)
        controls_layout.addWidget(self.refresh_all_btn)
        controls_layout.addWidget(self.play_all_btn)
        sidebar_layout.addLayout(controls_layout)
        
        # Etichetta per l'elenco feed
        feeds_label = QLabel("Feed Disponibili:")
        feeds_label.setStyleSheet(f"color: {Theme.TEXT_PRIMARY}; font-weight: bold; margin-top: 8px;")
        sidebar_layout.addWidget(feeds_label)
        
        # Lista dei feed
        self.tab_widget = QListWidget()
        self.tab_widget.setStyleSheet(f"""
            QListWidget {{
                background: {Theme.SURFACE};
                border: 1px solid {Theme.BORDER};
                border-radius: 6px;
                color: {Theme.TEXT_PRIMARY};
                font-family: {Theme.FONT_FAMILY};
                font-size: 14px;
                padding: 6px;
            }}
            QListWidget::item {{
                padding: 8px;
                border-radius: 4px;
            }}
            QListWidget::item:selected {{
                background: {Theme.PRIMARY};
                color: white;
            }}
            QListWidget::item:hover {{
                background: {Theme.SURFACE_HOVER};
            }}
        """)
        self.tab_widget.itemClicked.connect(self.show_feed_videos)
        self.tab_widget.setContextMenuPolicy(Qt.CustomContextMenu)
        self.tab_widget.customContextMenuRequested.connect(self.show_feed_context_menu)
        sidebar_layout.addWidget(self.tab_widget)
        
        # Area principale destra per i video
        self.main_content_widget = QWidget()
        main_content_layout = QVBoxLayout(self.main_content_widget)
        main_content_layout.setContentsMargins(16, 16, 16, 16)
        main_content_layout.setSpacing(12)
        
        # Titolo area video
        self.videos_title_label = QLabel("Feed Unificato")
        self.videos_title_label.setStyleSheet(f"color: {Theme.TEXT_PRIMARY}; font-size: 18px; font-weight: bold;")
        main_content_layout.addWidget(self.videos_title_label)
        
        # Area di scorrimento per la griglia dei video
        self.videos_scroll = QScrollArea()
        self.videos_scroll.setWidgetResizable(True)
        self.videos_scroll.setStyleSheet(f"background: {Theme.BACKGROUND}; border: none;")
        self.videos_scroll.verticalScrollBar().valueChanged.connect(self.on_scroll)
        
        # Griglia dei video per il feed selezionato o la vista unificata
        self.video_grid_container = QWidget()
        self.video_grid_layout = QGridLayout(self.video_grid_container)
        self.video_grid_layout.setSpacing(16)
        self.video_grid_layout.setContentsMargins(10, 10, 10, 10)
        self.videos_scroll.setWidget(self.video_grid_container)
        
        main_content_layout.addWidget(self.videos_scroll)
        
        # Aggiungi i widget al layout principale
        main_layout.addWidget(self.sidebar_widget)
        main_layout.addWidget(self.main_content_widget)
        
        # Segnaposto per stato vuoto
        self.placeholder_label = QLabel("Seleziona un feed o visualizza tutti i video nel Feed Unificato")
        self.placeholder_label.setAlignment(Qt.AlignCenter)
        self.placeholder_label.setStyleSheet(f"color: {Theme.TEXT_SECONDARY}; font-style: italic; font-size: 16px;")
        self.video_grid_layout.addWidget(self.placeholder_label, 0, 0, 1, -1)

    def refresh_feeds(self):
        """Aggiorna l'elenco dei feed."""
        if self.refresh_in_progress:
            logger.info("📡 [DEBUG] Aggiornamento già in corso, salto")
            return
        self.refresh_in_progress = True
        logger.info("📡 [DEBUG] Aggiornamento elenco feed")
        try:
            self.tab_widget.clear()
            # Aggiungi "Feed Unificato" come primo elemento
            unified_item = QListWidgetItem("Feed Unificato")
            unified_item.setData(Qt.UserRole, -1)
            self.tab_widget.addItem(unified_item)
            # Aggiungi tutti i feed RSS
            feeds = self.rss_service.get_feeds()
            logger.info(f"📡 [DEBUG] Trovati {len(feeds)} feed")
            for feed in feeds:
                item = QListWidgetItem(feed.title)
                item.setData(Qt.UserRole, feed.id)
                self.tab_widget.addItem(item)
                logger.info(f"📡 [DEBUG] Aggiunto feed: {feed.title} (ID: {feed.id})")
            # Seleziona il primo elemento se disponibile
            if self.tab_widget.count() > 0:
                self.tab_widget.setCurrentRow(0)
                QTimer.singleShot(50, lambda: self.show_feed_videos(self.tab_widget.item(0)))
            self.feed_refreshed.emit()
        except Exception as e:
            logger.error(f"❌ [DEBUG] Errore nell'aggiornamento dei feed: {e}")
        finally:
            self.refresh_in_progress = False

    def show_feed_videos(self, item):
        """Mostra i video per il feed selezionato o il feed unificato."""
        if not item:
            return
        if self.is_loading:
            return
        feed_id = item.data(Qt.UserRole)
        feed_name = item.text()
        logger.info(f"📡 [DEBUG] Mostra video per: {feed_name} (ID: {feed_id})")
        
        # Aggiorna il titolo dell'area video
        if feed_id == -1:
            self.videos_title_label.setText("Feed Unificato - Tutti i video")
        else:
            self.videos_title_label.setText(f"Feed: {feed_name}")
        
        # Pulisci i video precedenti e mostra lo stato di caricamento
        self.clear_video_grid()
        self.show_loading_state()
        try:
            if feed_id == -1:
                self.load_unified_feed()
            else:
                self.load_feed_videos(feed_id)
        except Exception as e:
            logger.error(f"❌ [DEBUG] Errore nel caricamento dei video: {e}")
            self.show_error_state(f"Errore nel caricamento dei video: {str(e)}")

    def clear_video_grid(self):
        """Pulisce completamente la griglia dei video."""
        for i in reversed(range(self.video_grid_layout.count())):
            widget = self.video_grid_layout.itemAt(i).widget()
            if widget:
                widget.setParent(None)
                widget.deleteLater()

    def show_loading_state(self):
        """Mostra lo stato di caricamento nella griglia dei video."""
        self.clear_video_grid()
        loading_label = QLabel("Caricamento video...")
        loading_label.setAlignment(Qt.AlignCenter)
        loading_label.setStyleSheet(f"color: {Theme.TEXT_SECONDARY}; font-style: italic; font-size: 16px;")
        self.video_grid_layout.addWidget(loading_label, 0, 0, 1, -1)

    def show_error_state(self, message):
        """Mostra lo stato di errore nella griglia dei video."""
        self.clear_video_grid()
        error_label = QLabel(message)
        error_label.setAlignment(Qt.AlignCenter)
        error_label.setStyleSheet(f"color: {Theme.DANGER}; font-size: 14px;")
        self.video_grid_layout.addWidget(error_label, 0, 0, 1, -1)
        self.is_loading = False

    def show_empty_state(self, message):
        """Mostra lo stato vuoto nella griglia dei video."""
        self.clear_video_grid()
        empty_label = QLabel(message)
        empty_label.setAlignment(Qt.AlignCenter)
        empty_label.setStyleSheet(f"color: {Theme.TEXT_SECONDARY}; font-style: italic; font-size: 14px;")
        self.video_grid_layout.addWidget(empty_label, 0, 0, 1, -1)
        self.is_loading = False

    def load_unified_feed(self):
        """Carica tutti i video da tutti i feed (feed unificato)."""
        if self.is_loading:
            return
        self.is_loading = True
        self.current_offset = 0
        logger.info("📡 [DEBUG] Caricamento feed unificato")
        def load_task():
            try:
                videos = self.rss_service.get_all_feed_items_as_videos(limit=30, offset=0)
                logger.info(f"📡 [DEBUG] Feed unificato caricato {len(videos)} video")
                QMetaObject.invokeMethod(self, "_update_unified_feed",
                                       Qt.QueuedConnection,
                                       Q_ARG(list, videos))
            except Exception as e:
                logger.error(f"❌ [DEBUG] Errore nel caricamento del feed unificato: {e}")
                QMetaObject.invokeMethod(self, "_handle_load_error",
                                       Qt.QueuedConnection,
                                       Q_ARG(str, f"Impossibile caricare il feed unificato: {str(e)}"))
        self.run_in_background(load_task)

    @pyqtSlot(list)
    def _update_unified_feed(self, videos):
        """Metodo thread-safe per aggiornare il feed unificato."""
        try:
            logger.info(f"📡 [DEBUG] Aggiornamento feed unificato con {len(videos)} video")
            self.all_videos = videos
            if not videos:
                logger.warning("📡 [DEBUG] Nessun video trovato nel feed unificato")
                self.show_empty_state("Nessun video trovato in alcun feed. Prova ad aggiornare o aggiungere nuovi feed RSS.")
                return
            self.sort_videos()
            self.update_video_grid()
            self.main_window.statusBar().showMessage(f"Caricati {len(self.all_videos)} video da tutti i feed")
        except Exception as e:
            logger.error(f"❌ [DEBUG] Errore nell'aggiornamento del feed unificato: {e}")
            self.show_error_state(f"Errore nella visualizzazione dei video: {str(e)}")
        finally:
            self.is_loading = False

    def load_feed_videos(self, feed_id):
        """Carica i video per un feed specifico."""
        if self.is_loading:
            return
        self.is_loading = True
        self.current_offset = 0
        logger.info(f"📡 [DEBUG] Caricamento video per feed ID: {feed_id}")
        def load_task():
            try:
                videos = self.rss_service.get_feed_videos(feed_id, limit=30, offset=0)
                logger.info(f"📡 [DEBUG] Feed {feed_id} caricato {len(videos)} video")
                QMetaObject.invokeMethod(self, "_update_feed_videos",
                                       Qt.QueuedConnection,
                                       Q_ARG(list, videos),
                                       Q_ARG(int, feed_id))
            except Exception as e:
                logger.error(f"❌ [DEBUG] Errore nel caricamento dei video del feed: {e}")
                QMetaObject.invokeMethod(self, "_handle_load_error",
                                       Qt.QueuedConnection,
                                       Q_ARG(str, f"Impossibile caricare i video del feed: {str(e)}"))
        self.run_in_background(load_task)

    @pyqtSlot(list, int)
    def _update_feed_videos(self, videos, feed_id):
        """Metodo thread-safe per aggiornare i video del feed."""
        try:
            logger.info(f"📡 [DEBUG] Aggiornamento feed {feed_id} con {len(videos)} video")
            self.all_videos = videos
            if not videos:
                current_item = self.tab_widget.currentItem()
                feed_name = current_item.text() if current_item else "Sconosciuto"
                self.show_empty_state(f"Nessun video trovato in '{feed_name}'. Prova ad aggiornare il feed.")
                return
            self.update_video_grid()
            self.main_window.statusBar().showMessage(f"Caricati {len(self.all_videos)} video dal feed")
        except Exception as e:
            logger.error(f"❌ [DEBUG] Errore nell'aggiornamento dei video del feed: {e}")
            self.show_error_state(f"Errore nella visualizzazione dei video: {str(e)}")
        finally:
            self.is_loading = False

    @pyqtSlot(str)
    def _handle_load_error(self, error_msg):
        """Metodo thread-safe per gestire gli errori di caricamento."""
        logger.error(f"❌ [DEBUG] Errore di caricamento: {error_msg}")
        self.show_error_state(error_msg)

    def update_video_grid(self):
        """Aggiorna la griglia dei video con i video attuali."""
        self.clear_video_grid()
        if not self.all_videos:
            self.show_empty_state("Nessun video disponibile.")
            return
        logger.info(f"📡 [DEBUG] Creazione griglia video con {len(self.all_videos)} video")
        # Visualizza i video in griglia
        columns = 2
        for idx, video in enumerate(self.all_videos):
            try:
                logger.debug(f"📡 [DEBUG] Creazione VideoCard per: {video.title}")
                card = VideoCard(video, main_window=self.main_window)
                # Connetti i segnali
                card.play_requested.connect(self.video_card_play_requested.emit)
                card.add_to_queue_requested.connect(self.add_to_queue_requested.emit)
                # Menu contestuale tasto destro (gestito da VideoCard)
                card.setContextMenuPolicy(Qt.CustomContextMenu)
                # Aggiungi alla griglia
                row = idx // columns
                col = idx % columns
                self.video_grid_layout.addWidget(card, row, col)
            except Exception as e:
                logger.error(f"❌ [DEBUG] Errore nella creazione di VideoCard per '{video.title}': {e}")
                error_label = QLabel(f"Errore: {video.title}")
                error_label.setStyleSheet(f"color: {Theme.DANGER}; padding: 10px;")
                row = idx // columns
                col = idx % columns
                self.video_grid_layout.addWidget(error_label, row, col)

    def sort_videos(self):
        """Ordina i video per data di pubblicazione."""
        try:
            self.all_videos.sort(
                key=lambda x: x.published if hasattr(x, 'published') and x.published else x.updated,
                reverse=self.sort_newest_first
            )
        except Exception as e:
            logger.error(f"Errore nell'ordinamento dei video: {e}")

    def on_scroll(self, value):
        """Gestisce lo scorrimento infinito per la griglia dei video."""
        scroll_bar = self.videos_scroll.verticalScrollBar()
        if not self.is_loading and scroll_bar.value() >= scroll_bar.maximum() - 100:
            self.load_more_videos()

    def load_more_videos(self):
        """Carica più video per lo scorrimento infinito."""
        if self.is_loading:
            return
        self.is_loading = True
        try:
            current_item = self.tab_widget.currentItem()
            if not current_item:
                self.is_loading = False
                return
            feed_id = current_item.data(Qt.UserRole)
            def load_task():
                try:
                    new_videos = []
                    if feed_id == -1:
                        new_videos = self.rss_service.get_all_feed_items_as_videos(
                            limit=30,
                            offset=self.current_offset
                        )
                    else:
                        new_videos = self.rss_service.get_feed_videos(
                            feed_id,
                            limit=30,
                            offset=self.current_offset
                        )
                    if not new_videos:
                        logger.info("Nessun altro video da caricare.")
                        self.is_loading = False
                        return
                    # Aggiornamento thread-safe
                    QMetaObject.invokeMethod(self, "_append_videos",
                                           Qt.QueuedConnection,
                                           Q_ARG(list, new_videos),
                                           Q_ARG(int, feed_id))
                except Exception as e:
                    logger.error(f"Errore nel caricamento di ulteriori video: {e}")
                    QMetaObject.invokeMethod(self, "_handle_load_more_error",
                                           Qt.QueuedConnection,
                                           Q_ARG(str, str(e)))
            self.run_in_background(load_task)
        except Exception as e:
            logger.error(f"Errore in load_more_videos: {e}")
            self.is_loading = False

    @pyqtSlot(list, int)
    def _append_videos(self, new_videos, feed_id):
        """Metodo thread-safe per aggiungere video."""
        try:
            self.all_videos.extend(new_videos)
            if feed_id == -1:
                self.sort_videos()
            # Aggiungi solo i nuovi video, non tutti i video
            columns = 2
            start_idx = len(self.all_videos) - len(new_videos)
            for idx, video in enumerate(new_videos, start=start_idx):
                try:
                    card = VideoCard(video, main_window=self.main_window)
                    card.play_requested.connect(self.video_card_play_requested.emit)
                    card.add_to_queue_requested.connect(self.add_to_queue_requested.emit)
                    card.setContextMenuPolicy(Qt.CustomContextMenu)
                    row = idx // columns
                    col = idx % columns
                    self.video_grid_layout.addWidget(card, row, col)
                except Exception as e:
                    logger.error(f"❌ [DEBUG] Errore nella creazione di VideoCard per '{video.title}': {e}")
                    error_label = QLabel(f"Errore: {video.title}")
                    error_label.setStyleSheet(f"color: {Theme.DANGER}; padding: 10px;")
                    row = idx // columns
                    col = idx % columns
                    self.video_grid_layout.addWidget(error_label, row, col)
            self.current_offset += len(new_videos)
        except Exception as e:
            logger.error(f"Errore nell'aggiunta dei video: {e}")
        finally:
            self.is_loading = False

    @pyqtSlot(str)
    def _handle_load_more_error(self, error_msg):
        """Metodo thread-safe per gestire gli errori di caricamento aggiuntivi."""
        QMessageBox.warning(self, "Errore", f"Impossibile caricare ulteriori video: {error_msg}")
        self.is_loading = False

    def refresh_all_feeds(self):
        """Aggiorna tutti i feed e aggiorna la vista corrente."""
        if self.refresh_in_progress:
            return
        def refresh_task():
            try:
                success = self.rss_service.refresh_all_feeds()
                if not success:
                    raise RuntimeError("Impossibile aggiornare i feed.")
                self.refresh_complete.emit()
            except Exception as e:
                logger.error(f"Errore nell'aggiornamento dei feed: {e}")
                self.refresh_error.emit(str(e))
        self.run_in_background(refresh_task)

    @pyqtSlot()
    def _handle_refresh_complete(self):
        """Metodo thread-safe per gestire il completamento dell'aggiornamento."""
        current_item = self.tab_widget.currentItem()
        if current_item:
            self.show_feed_videos(current_item)
        self.main_window.statusBar().showMessage("Tutti i feed aggiornati con successo")
        self.feed_refreshed.emit()

    @pyqtSlot(str)
    def _handle_refresh_error(self, error_msg):
        """Metodo thread-safe per gestire gli errori di aggiornamento."""
        QMessageBox.warning(self, "Errore", f"Impossibile aggiornare i feed: {error_msg}")

    def play_all_videos(self):
        """Riproduce tutti i video nella vista corrente."""
        current_item = self.tab_widget.currentItem()
        if not current_item:
            QMessageBox.warning(self, "Avviso", "Nessun feed selezionato")
            return
        if not self.all_videos:
            QMessageBox.warning(self, "Avviso", "Nessun video in questo feed")
            return
        if hasattr(self.main_window, 'vlc_player') and hasattr(self.main_window.vlc_player, 'queue_manager'):
            self.main_window.vlc_player.queue_manager.clear_queue()
        for video in self.all_videos:
            self.add_to_queue_requested.emit(video)
        if self.all_videos:
            self.video_card_play_requested.emit(self.all_videos[0])
        self.main_window.statusBar().showMessage(f"Riproduzione di tutti i {len(self.all_videos)} video")

    def show_feed_context_menu(self, pos):
        """Mostra il menu contestuale per i feed."""
        item = self.tab_widget.itemAt(pos)
        if not item or item.data(Qt.UserRole) == -1:
            return
        menu = QMenu(self)
        rename_action = menu.addAction("Rinomina Feed")
        rename_action.triggered.connect(lambda: self.rename_feed(item))
        delete_action = menu.addAction("Elimina Feed")
        delete_action.triggered.connect(lambda: self.delete_feed(item))
        refresh_action = menu.addAction("Aggiorna Feed")
        refresh_action.triggered.connect(lambda: self.refresh_feed(item))
        menu.exec_(self.tab_widget.mapToGlobal(pos))

    def rename_feed(self, item):
        """Rinomina il feed selezionato."""
        feed_name = item.text()
        new_name, ok = QInputDialog.getText(
            self,
            "Rinomina Feed",
            "Inserisci nuovo nome:",
            text=feed_name
        )
        if ok and new_name and new_name != feed_name:
            try:
                feed_id = item.data(Qt.UserRole)
                self.rss_service.update_feed_name(feed_id, new_name)
                self.refresh_feeds()
                self.main_window.statusBar().showMessage(f"Feed rinominato in: {new_name}")
            except Exception as e:
                logger.error(f"❌ [DEBUG] Errore nella rinomina del feed: {e}")
                QMessageBox.warning(self, "Errore", f"Impossibile rinominare il feed: {str(e)}")

    def delete_feed(self, item):
        """Elimina il feed selezionato dal menu contestuale."""
        self.tab_widget.setCurrentItem(item)
        self.delete_selected_feed()

    def refresh_feed(self, item):
        """Aggiorna un feed specifico."""
        feed_id = item.data(Qt.UserRole)
        feed_name = item.text()
        def refresh_task():
            try:
                success = self.rss_service.refresh_feed(feed_id)
                if not success:
                    raise RuntimeError(f"Impossibile aggiornare il feed: {feed_name}")
                QMetaObject.invokeMethod(self, "_handle_feed_refresh_complete",
                                       Qt.QueuedConnection,
                                       Q_ARG(int, feed_id),
                                       Q_ARG(str, feed_name))
            except Exception as e:
                logger.error(f"Errore nell'aggiornamento del feed: {e}")
                self.refresh_error.emit(str(e))
        self.run_in_background(refresh_task)

    @pyqtSlot(int, str)
    def _handle_feed_refresh_complete(self, feed_id, feed_name):
        """Metodo thread-safe per gestire il completamento dell'aggiornamento del feed."""
        current_item = self.tab_widget.currentItem()
        if current_item and current_item.data(Qt.UserRole) == feed_id:
            self.show_feed_videos(current_item)
        self.main_window.statusBar().showMessage(f"Feed aggiornato: {feed_name}")
        self.feed_refreshed.emit()

    def add_new_feed(self):
        """Aggiunge un nuovo feed RSS."""
        url, ok = QInputDialog.getText(
            self,
            "Aggiungi Feed RSS",
            "Inserisci URL del feed RSS:",
            text="https://"
        )
        if ok and url:
            try:
                if not url.startswith(("http://", "https://")):
                    QMessageBox.warning(self, "Errore", "L'URL deve iniziare con http:// o https://")
                    return
                logger.info(f"📡 [DEBUG] Aggiunta nuovo feed: {url}")
                if self.rss_service.add_feed(url):
                    self.refresh_feeds()
                    self.main_window.statusBar().showMessage(f"Aggiunto nuovo feed RSS: {url}")
                    logger.info(f"📡 [DEBUG] Feed aggiunto con successo: {url}")
                else:
                    QMessageBox.warning(self, "Errore", f"Impossibile aggiungere il feed RSS: {url}")
                    logger.error(f"❌ [DEBUG] Impossibile aggiungere il feed: {url}")
            except Exception as e:
                logger.error(f"❌ [DEBUG] Errore nell'aggiunta del feed: {e}")
                QMessageBox.warning(self, "Errore", f"Impossibile aggiungere il feed: {str(e)}")

    def delete_selected_feed(self):
        """Elimina il feed selezionato."""
        current_item = self.tab_widget.currentItem()
        if not current_item:
            QMessageBox.warning(self, "Avviso", "Nessun feed selezionato")
            return
        if current_item.data(Qt.UserRole) == -1:
            QMessageBox.warning(self, "Avviso", "Impossibile eliminare il Feed Unificato")
            return
        feed_name = current_item.text()
        reply = QMessageBox.question(
            self,
            "Elimina Feed",
            f"Sei sicuro di voler eliminare il feed '{feed_name}'?",
            QMessageBox.Yes | QMessageBox.No
        )
        if reply == QMessageBox.Yes:
            try:
                feed_id = current_item.data(Qt.UserRole)
                if self.rss_service.remove_feed(feed_id):
                    self.refresh_feeds()
                    self.main_window.statusBar().showMessage(f"Feed eliminato: {feed_name}")
                else:
                    QMessageBox.warning(self, "Errore", f"Impossibile eliminare il feed: {feed_name}")
            except Exception as e:
                logger.error(f"❌ [DEBUG] Errore nell'eliminazione del feed: {e}")
                QMessageBox.warning(self, "Errore", f"Impossibile eliminare il feed: {str(e)}")

    def add_video_to_playlist(self, video):
        """Aggiunge un video a una playlist."""
        if not hasattr(self.main_window, 'playlist_panel'):
            QMessageBox.warning(self, "Avviso", "Funzionalità playlist non disponibile")
            return
        # In un'implementazione reale, mostra una finestra di dialogo per selezionare una playlist
        self.main_window.statusBar().showMessage(f"Funzionalità aggiungi a playlist per: {video.title}")

    def run_in_background(self, task):
        """Esegue un'attività in background utilizzando QRunnable."""
        class Worker(QRunnable):
            def __init__(self, task):
                super().__init__()
                self.task = task
            def run(self):
                self.task()
        worker = Worker(task)
        QThreadPool.globalInstance().start(worker)

    def debug_rss_service(self):
        """Metodo di debug per verificare lo stato del servizio RSS."""
        try:
            feeds = self.rss_service.get_feeds()
            print(f"📡 [DEBUG] Trovati {len(feeds)} feed:")
            for feed in feeds:
                print(f"  - {feed.title} (ID: {feed.id})")
            # Test feed unificato
            videos = self.rss_service.get_all_feed_items_as_videos(limit=5, offset=0)
            print(f"📡 [DEBUG] Il feed unificato ha {len(videos)} video")
            # Test feed individuali
            for feed in feeds:
                feed_videos = self.rss_service.get_feed_videos(feed.id, limit=5, offset=0)
                print(f"📡 [DEBUG] Il feed '{feed.title}' ha {len(feed_videos)} video")
        except Exception as e:
            print(f"❌ [DEBUG] Errore servizio RSS: {e}")
