"""
Impostazioni, costanti e configurazioni dell'applicazione - Versione avanzata.
Questo modulo centralizza tutte le impostazioni dell'applicazione, i percorsi,
le configurazioni API, le impostazioni predefinite dell'interfaccia utente e le politiche di sicurezza.
Include anche funzioni di utilità per la gestione dei percorsi e la validazione dell'ambiente.
"""
import os
import time
import logging
import platform
import sys
import threading
import json
from collections import deque
from pathlib import Path
from typing import Dict, List, Optional, Tuple, Any
from datetime import datetime, timedelta

# --- Metadati dell'Applicazione ---
APP_NAME: str = "PurpleTube Player"
APP_AUTHOR: str = "Angelo Massaro"
APP_LICENSE: str = "GNU GPL v3"
APP_DESCRIPTION: str = (
    "Un lettore video decentralizzato per PeerTube, "
    "che promuove la libertà digitale e l'accesso etico ai media."
)
APP_WEBSITE: str = "https://boostmedia.it/purpletube"
APP_REPO: str = "https://github.com/boostmedia-aps/purpletube-player"

# --- Gestione dei Percorsi ---
def get_data_dir() -> Path:
    """Restituisce la directory dati appropriata in base al sistema operativo."""
    if os.name == 'nt':  # Windows
        return Path(os.getenv('APPDATA', '')) / "PurpleTube"
    elif sys.platform == 'darwin':  # macOS
        return Path.home() / "Library" / "Application Support" / "PurpleTube"
    else:  # Linux/Unix
        return Path.home() / ".purpletube_player"

DATA_DIR: Path = get_data_dir()
DB_PATH: Path = DATA_DIR / "database.db"
LOG_PATH: Path = DATA_DIR / "app.log"
CACHE_DIR: Path = DATA_DIR / "cache"
THUMBNAIL_CACHE_DIR: Path = CACHE_DIR / "thumbnails"
DOWNLOAD_DIR: Path = Path.home() / "Videos" / "PurpleTube"
SETTINGS_FILE: Path = DATA_DIR / "user_settings.json"

# Assicura che le directory esistano e imposta i permessi
def ensure_directories() -> None:
    """Crea tutte le directory richieste e imposta i permessi appropriati."""
    DATA_DIR.mkdir(parents=True, exist_ok=True)
    CACHE_DIR.mkdir(parents=True, exist_ok=True)
    THUMBNAIL_CACHE_DIR.mkdir(parents=True, exist_ok=True)
    DOWNLOAD_DIR.mkdir(parents=True, exist_ok=True)
    if os.name != 'nt':  # Non Windows
        DATA_DIR.chmod(0o700)
        CACHE_DIR.chmod(0o700)
        THUMBNAIL_CACHE_DIR.chmod(0o700)

ensure_directories()

# --- Gestione delle Impostazioni Utente ---
class SettingsManager:
    """
    Gestisce il caricamento, salvataggio e accesso alle impostazioni dell'utente.
    """

    # Default settings structure
    DEFAULT_SETTINGS = {
        "default_player": "Browser Predefinito",
        "theme": "dark",
        "auto_play": False,
        "volume": 70,
        "playback_speed": 1.0,
        "subtitle_language": "en",
        "video_quality": "bilanciato",
        "max_search_results": 50,
        "enable_animations": True,
        "thumbnail_quality": "medium",
        "download_directory": str(Path.home() / "Videos" / "PurpleTube"),
        "language": "en",
        "region": "worldwide",
        "nsfw_filter": False,
        "left_click_action": "use_default"  # New setting
    }

    @classmethod
    def load_settings(cls) -> Dict[str, Any]:
        """Carica le impostazioni dal file. Se il file non esiste, restituisce le impostazioni predefinite."""
        try:
            if SETTINGS_FILE.exists():
                with open(SETTINGS_FILE, 'r', encoding='utf-8') as f:
                    loaded_settings = json.load(f)
                    # Merge with default settings to ensure all keys exist
                    settings = cls.DEFAULT_SETTINGS.copy()
                    settings.update(loaded_settings)
                    return settings
        except Exception as e:
            logging.error(f"Errore nel caricamento delle impostazioni: {e}")
        return cls.DEFAULT_SETTINGS.copy()

    @classmethod
    def save_settings(cls, settings: Dict[str, Any]) -> None:
        """Salva le impostazioni nel file."""
        try:
            # Ensure directory exists
            SETTINGS_FILE.parent.mkdir(parents=True, exist_ok=True)

            with open(SETTINGS_FILE, 'w', encoding='utf-8') as f:
                json.dump(settings, f, indent=4, ensure_ascii=False)
            logging.info("✅ Impostazioni salvate correttamente")

            # Reload settings after saving to update runtime config
            cls._update_runtime_config()
        except Exception as e:
            logging.error(f"Errore nel salvataggio delle impostazioni: {e}")

    @classmethod
    def get_setting(cls, key: str, default: Any = None) -> Any:
        """Restituisce un'impostazione specifica."""
        settings = cls.load_settings()
        return settings.get(key, default)

    @classmethod
    def set_setting(cls, key: str, value: Any) -> None:
        """Imposta un'impostazione specifica."""
        settings = cls.load_settings()
        settings[key] = value
        cls.save_settings(settings)

    @classmethod
    def get_default_player(cls) -> str:
        """Restituisce il player predefinito corrente."""
        settings = cls.load_settings()
        return settings.get("default_player", "Browser Predefinito")

    @classmethod
    def set_default_player(cls, player_name: str) -> None:
        """Imposta il player predefinito e aggiorna il file delle impostazioni."""
        settings = cls.load_settings()
        settings["default_player"] = player_name
        cls.save_settings(settings)
        logging.info(f"🔄 [DEBUG] Default player impostato: {player_name}")

    @classmethod
    def get_left_click_action(cls) -> str:
        """Restituisce l'azione del click sinistro calcolata in base alle impostazioni."""
        settings = cls.load_settings()
        left_click_action = settings.get("left_click_action", "use_default")

        if left_click_action == "use_default":
            # Calcola l'azione predefinita in base al player predefinito
            default_player = settings.get("default_player", "Browser Predefinito")
            return "vlc" if default_player == "VLC" else "browser"
        else:
            return left_click_action

    @classmethod
    def set_left_click_action(cls, action: str) -> None:
        """Imposta l'azione del click sinistro."""
        settings = cls.load_settings()
        settings["left_click_action"] = action
        cls.save_settings(settings)
        logging.info(f"🔄 [DEBUG] Left-click action impostata: {action}")

    @classmethod
    def initialize_settings(cls) -> None:
        """Inizializza le impostazioni al primo avvio dell'applicazione."""
        if not SETTINGS_FILE.exists():
            logging.info("🎯 Inizializzazione impostazioni per il primo avvio")
            cls.save_settings(cls.DEFAULT_SETTINGS)
            cls._update_runtime_config()

    @classmethod
    def _update_runtime_config(cls) -> None:
        """Aggiorna le configurazioni di runtime in base alle impostazioni correnti."""
        try:
            settings = cls.load_settings()

            # Aggiorna PlayerConfig con l'azione del click sinistro calcolata
            PlayerConfig.LEFT_CLICK_ACTION = cls.get_left_click_action()

            # Aggiorna altre configurazioni di PlayerConfig
            PlayerConfig.ENABLE_AUTO_PLAY = settings.get("auto_play", False)
            PlayerConfig.DEFAULT_VOLUME = settings.get("volume", 70)
            PlayerConfig.DEFAULT_PLAYBACK_SPEED = settings.get("playback_speed", 1.0)
            PlayerConfig.DEFAULT_SUBTITLE_LANGUAGE = settings.get("subtitle_language", "en")

            # Aggiorna UIConfig
            UIConfig.DEFAULT_THEME = settings.get("theme", "dark")
            UIConfig.ENABLE_ANIMATIONS = settings.get("enable_animations", True)

            logging.info(f"🔄 Configurazioni di runtime aggiornate - LEFT_CLICK_ACTION: {PlayerConfig.LEFT_CLICK_ACTION}")

        except Exception as e:
            logging.error(f"❌ Errore nell'aggiornamento delle configurazioni di runtime: {e}")

    @classmethod
    def reset_to_defaults(cls) -> None:
        """Ripristina tutte le impostazioni ai valori predefiniti."""
        cls.save_settings(cls.DEFAULT_SETTINGS)
        logging.info("🔄 Impostazioni ripristinate ai valori predefiniti")

    @classmethod
    def export_settings(cls, export_path: Path) -> bool:
        """Esporta le impostazioni in un file."""
        try:
            settings = cls.load_settings()
            with open(export_path, 'w', encoding='utf-8') as f:
                json.dump(settings, f, indent=4, ensure_ascii=False)
            logging.info(f"✅ Impostazioni esportate in: {export_path}")
            return True
        except Exception as e:
            logging.error(f"❌ Errore nell'esportazione delle impostazioni: {e}")
            return False

    @classmethod
    def import_settings(cls, import_path: Path) -> bool:
        """Importa le impostazioni da un file."""
        try:
            with open(import_path, 'r', encoding='utf-8') as f:
                imported_settings = json.load(f)

            # Merge con le impostazioni correnti
            current_settings = cls.load_settings()
            current_settings.update(imported_settings)
            cls.save_settings(current_settings)

            logging.info(f"✅ Impostazioni importate da: {import_path}")
            return True
        except Exception as e:
            logging.error(f"❌ Errore nell'importazione delle impostazioni: {e}")
            return False

    @classmethod
    def get_all_settings(cls) -> Dict[str, Any]:
        """Restituisce tutte le impostazioni correnti."""
        return cls.load_settings()

    @classmethod
    def trigger_rickroll(cls) -> None:
        """
        Easter Egg: Rickroll l'utente aprendo il famoso video in modo divertente.
        """
        rickroll_url = "https://media.geekwisdom.org/videos/watch/32e25713-c01f-4651-ad28-4d3fdcbec5ea"
        logging.info("🎵 Easter Egg attivato: Rickroll in arrivo! Never gonna give you up! 🎵")
        PlayerConfig.open_with_default_player(rickroll_url)

# Inizializza le impostazioni al primo avvio
SettingsManager.initialize_settings()

# --- Limitazione del Tasso di Chiamate API ---
class APIRateLimiter:
    MAX_CALLS = 5000
    TIME_WINDOW = 3600  # 1 ora

    def __init__(self):
        self.call_times = deque()
        self.lock = threading.Lock()

    def increment(self) -> bool:
        current_time = time.time()
        with self.lock:
            while self.call_times and (current_time - self.call_times[0] > self.TIME_WINDOW):
                self.call_times.popleft()
            if len(self.call_times) >= self.MAX_CALLS:
                logging.error(f"Limite di chiamate API superato: {len(self.call_times)}/{self.MAX_CALLS} nell'ultima ora")
                return False
            self.call_times.append(current_time)
            logging.debug(f"Chiamata API {len(self.call_times)}/{self.MAX_CALLS} nell'ultima ora")
            return True

    def reset(self):
        with self.lock:
            self.call_times.clear()

    def remaining(self) -> int:
        current_time = time.time()
        with self.lock:
            while self.call_times and (current_time - self.call_times[0] > self.TIME_WINDOW):
                self.call_times.popleft()
            return max(0, self.MAX_CALLS - len(self.call_times))

API_RATE_LIMITER = APIRateLimiter()

# --- Impostazioni API ---
SEPIASEARCH_BASE_URL: str = "https://sepiasearch.org/api/v1/search/videos"
PEERTUBE_API_TIMEOUT: int = 10
DEFAULT_SEARCH_COUNT: int = 50
MAX_SEARCH_RESULTS: int = 1000
API_REQUEST_TIMEOUT: int = 10
API_USER_AGENT: str = f"{APP_NAME} (Python/{platform.python_version()})"

# --- Configurazione del Limite di Tasso ---
MAX_API_CALLS: int = 5000
TIME_WINDOW: int = 3600
BURST_LIMIT: int = 100
BURST_WINDOW: int = 60

# --- Configurazione del Servizio di Ricerca ---
class SearchConfig:
    DEFAULT_SORT_BY: str = "-match"
    DEFAULT_LANGUAGE: Optional[str] = None
    DEFAULT_CATEGORY: Optional[str] = None
    DEFAULT_DURATION: Optional[str] = None
    DEFAULT_PUBLISHED_AFTER: Optional[str] = None
    SORT_OPTIONS: Dict[str, str] = {
        "Migliore Corrispondenza": "-match",
        "Più Recenti": "-publishedAt",
        "Meno Recenti": "publishedAt",
        "Più Visti": "-views",
        "Miglior Valutazione": "-likes",
        "In Tendenza": "-hot"
    }
    LANGUAGE_OPTIONS: Dict[str, str] = {
        "Qualsiasi Lingua": "", "Inglese": "en", "Spagnolo": "es", "Francese": "fr", "Tedesco": "de",
        "Italiano": "it", "Portoghese": "pt", "Russo": "ru", "Cinese": "zh", "Giapponese": "ja",
        "Arabo": "ar", "Hindi": "hi", "Bengalese": "bn", "Punjabi": "pa", "Turco": "tr",
        "Olandese": "nl", "Svedese": "sv", "Polacco": "pl", "Ucraino": "uk", "Vietnamita": "vi",
        "Coreano": "ko", "Indonesiano": "id", "Finlandese": "fi", "Norvegese": "no", "Danese": "da",
        "Ceco": "cs", "Ungherese": "hu", "Rumeno": "ro", "Greco": "el", "Thailandese": "th",
        "Ebraico": "he", "Persiano": "fa", "Urdu": "ur", "Filippino": "fil", "Malese": "ms",
        "Swahili": "sw"
    }
    CATEGORY_OPTIONS: Dict[str, str] = {
        "Qualsiasi Categoria": "", "Film": "2", "Veicoli": "3", "Arte": "4", "Sport": "5",
        "Viaggi": "6", "Gaming": "7", "Persone": "8", "Commedia": "9", "Intrattenimento": "10",
        "Notizie e Politica": "11", "Tutorial": "12", "Istruzione": "13", "Attivismo": "14",
        "Scienza e Tecnologia": "15", "Animali": "16", "Cibo": "18", "Musica": "1", "Documentario": "17",
        "Altro": "19"
    }
    DURATION_OPTIONS: Dict[str, str] = {
        "Qualsiasi Durata": "", "Breve (< 4 min)": "short", "Media (4-20 min)": "medium", "Lunga (> 20 min)": "long"
    }
    TRENDING_TIME_RANGES: Dict[str, str] = {
        "Ultime 24 Ore": "-last_24hours", "Ultimi 7 Giorni": "-last_7days",
        "Ultimi 30 Giorni": "-last_30days", "Ultimo Anno": "-last_year", "Tutti i Tempi": ""
    }
    MAX_PAGE_SIZE: int = 100
    MAX_QUERY_LENGTH: int = 256
    MAX_FILTER_VALUE_LENGTH: int = 64

# --- Impostazioni del Lettore ---
class PlayerConfig:
    DEFAULT_PLAYER: str = "Browser Predefinito"
    SUPPORTED_PLAYERS: List[str] = ["Browser Predefinito", "VLC", "IINA"]
    VLC_ARGS: List[str] = [
        "--verbose=0", "--no-xlib", "--network-caching=4000", "--file-caching=4000",
        "--live-caching=4000", "--sout-mux-caching=2000", "--clock-synchro=0", "--avcodec-threads=0",
        "--avcodec-hw=any", "--drop-late-frames", "--skip-frames", "--input-fast-seek",
        "--http-reconnect", "--codec=avcodec,none"
    ]
    VIDEO_QUALITY_PREFERENCES: Dict[str, List[str]] = {
        "velocità": ["worst[height<=720]", "worst[ext=mp4]/worst"],
        "bilanciato": ["best[height<=1080]/best[height<=720]", "best[ext=mp4]/best"],
        "qualità": ["best[height<=1440]/best[height<=1080]", "best"]
    }
    DEFAULT_VOLUME: int = 70
    DEFAULT_PLAYBACK_SPEED: float = 1.0
    DEFAULT_SUBTITLE_LANGUAGE: str = "en"
    ENABLE_AUTO_PLAY: bool = False

    # LEFT_CLICK_ACTION sarà aggiornata dinamicamente da SettingsManager
    LEFT_CLICK_ACTION: str = "browser"

    @classmethod
    def open_with_default_player(cls, url_or_file: str) -> None:
        """Apre un URL o file con il player predefinito."""
        player = SettingsManager.get_default_player()
        if player == "VLC":
            cls.open_with_vlc(url_or_file)
        else:
            cls.open_with_browser(url_or_file)

    @classmethod
    def open_with_vlc(cls, url_or_file: str) -> None:
        try:
            import subprocess
            system = platform.system()
            if system == "Windows":
                subprocess.Popen(['vlc', url_or_file])
            elif system == "Darwin":
                subprocess.Popen(['open', '-a', 'VLC', url_or_file])
            else:
                subprocess.Popen(['vlc', url_or_file])
            logging.info(f"Apertura con VLC: {url_or_file}")
        except Exception as e:
            logging.error(f"Errore nell'apertura con VLC: {e}")
            cls.open_with_browser(url_or_file)

    @classmethod
    def open_with_browser(cls, url: str) -> None:
        try:
            import webbrowser
            webbrowser.open(url)
            logging.info(f"Apertura con browser: {url}")
        except Exception as e:
            logging.error(f"Errore nell'apertura con browser: {e}")

    @classmethod
    def get_left_click_action(cls) -> str:
        """Restituisce l'azione del click sinistro corrente."""
        return cls.LEFT_CLICK_ACTION

    @classmethod
    def update_left_click_action(cls) -> None:
        """Aggiorna l'azione del click sinistro dalle impostazioni."""
        cls.LEFT_CLICK_ACTION = SettingsManager.get_left_click_action()
        logging.info(f"🔄 [DEBUG] LEFT_CLICK_ACTION aggiornata: {cls.LEFT_CLICK_ACTION}")

# Inizializza LEFT_CLICK_ACTION all'avvio
PlayerConfig.update_left_click_action()

# --- Impostazioni dell'Interfaccia Utente ---
class UIConfig:
    DEFAULT_WINDOW_SIZE: Tuple[int, int] = (1200, 800)
    MIN_WINDOW_SIZE: Tuple[int, int] = (800, 600)
    VIDEO_CARD_SIZE: Tuple[int, int] = (560, 360)
    ITEMS_PER_PAGE: int = 20
    DEFAULT_THEME: str = "dark"
    SUPPORTED_THEMES: List[str] = ["dark", "light", "system"]
    FONT_FAMILY: str = "'Inter', 'Segoe UI', 'Roboto', sans-serif"
    FONT_SIZE: int = 14
    ICON_THEME: str = "auto"
    ANIMATION_DURATION: int = 300
    ENABLE_ANIMATIONS: bool = True
    THUMBNAIL_LOAD_TIMEOUT: int = 10
    MIN_COLUMN_WIDTH: int = 300
    MAX_COLUMNS: int = 4
    GRID_SPACING: int = 16
    CARD_MARGIN: int = 8

# --- Configurazione del Logging ---
class LogConfig:
    LOG_LEVEL: int = logging.INFO
    LOG_FORMAT: str = '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
    MAX_LOG_SIZE: int = 1024 * 1024
    LOG_BACKUP_COUNT: int = 3
    LOG_ENCODING: str = 'utf-8'
    ENABLE_FILE_LOGGING: bool = True
    ENABLE_CONSOLE_LOGGING: bool = True

# --- Impostazioni del Database ---
class DBConfig:
    PRAGMA: Dict[str, str] = {
        "foreign_keys": "ON", "journal_mode": "WAL", "synchronous": "NORMAL",
        "temp_store": "MEMORY", "cache_size": "-2000"
    }
    MAX_CONNECTIONS: int = 5
    CONNECTION_TIMEOUT: int = 30
    ENABLE_BACKUPS: bool = True
    BACKUP_INTERVAL: int = 7

# --- Impostazioni di Sicurezza ---
class SecurityConfig:
    ENFORCE_HTTPS: bool = True
    ALLOWED_SCHEMES: List[str] = ["http", "https"]
    MAX_URL_LENGTH: int = 2048
    USER_AGENT: str = API_USER_AGENT
    CORS_ALLOW_ORIGIN: Optional[str] = None
    CSRF_PROTECTION: bool = True
    MAX_FILE_SIZE: int = 50 * 1024 * 1024

# --- Impostazioni di Rete ---
class NetworkConfig:
    REQUEST_TIMEOUT: int = 30
    MAX_RETRIES: int = 3
    RETRY_DELAY: float = 1.0
    RETRY_BACKOFF: float = 2.0
    PROXY_ENABLED: bool = False
    PROXY_URL: Optional[str] = None
    USER_AGENT: str = API_USER_AGENT
    ENABLE_COMPRESSION: bool = True

# --- Impostazioni della Cache ---
class CacheConfig:
    MAX_THUMBNAIL_CACHE_SIZE: int = 100
    THUMBNAIL_CACHE_TTL: int = 30
    MAX_MEMORY_CACHE_SIZE: int = 50
    ENABLE_DISK_CACHE: bool = True
    CACHE_CLEANUP_INTERVAL: int = 24
    SEARCH_CACHE_SIZE: int = 64
    TRENDING_CACHE_SIZE: int = 32
    FEED_CACHE_SIZE: int = 32
    CHANNEL_CACHE_SIZE: int = 32

# --- Valori Predefiniti ---
DEFAULT_LANGUAGE: str = "en"
DEFAULT_REGION: str = "worldwide"
DEFAULT_NSFW_FILTER: bool = False
DEFAULT_AUTO_PLAY: bool = False
DEFAULT_SUBTITLE_LANGUAGE: str = "en"
DEFAULT_PLAYBACK_SPEED: float = 1.0

# --- Feature Flags ---
class FeatureFlags:
    ENABLE_EXPERIMENTAL_FEATURES: bool = False
    ENABLE_ANALYTICS: bool = False
    ENABLE_AUTO_UPDATES: bool = True
    ENABLE_THUMBNAIL_PREVIEW: bool = True
    ENABLE_SMART_BUFFERING: bool = True
    ENABLE_ADVANCED_SEARCH: bool = True
    ENABLE_PLAYLIST_IMPORT_EXPORT: bool = True
    ENABLE_RSS_FEED_MANAGEMENT: bool = True

# --- Impostazioni di Esportazione/Importazione ---
class ExportImportConfig:
    SUPPORTED_FORMATS: List[str] = ["opml", "json"]
    MAX_EXPORT_SIZE: int = 50
    DEFAULT_EXPORT_FILENAME: str = "purpletube_export"
    BACKUP_INTERVAL: int = 7
    ENABLE_ENCRYPTED_BACKUPS: bool = False

# --- Costanti del Servizio di Ricerca ---
SEARCH_API_ENDPOINT: str = "https://sepiasearch.org/api/v1/search/videos"
TRENDING_API_ENDPOINT: str = "https://sepiasearch.org/api/v1/search/videos"
CHANNEL_API_ENDPOINT: str = "https://sepiasearch.org/api/v1/search/videos"
DEFAULT_PAGE_SIZE: int = 50
MAX_PAGE_SIZE: int = 100
DEFAULT_SORT: str = "-match"
DEFAULT_NSFW_FILTER: str = "false"
TIME_RANGES: Dict[str, str] = {
    "last_24hours": "-last_24hours", "last_7days": "-last_7days",
    "last_30days": "-last_30days", "last_year": "-last_year", "all_time": ""
}
DURATION_RANGES: Dict[str, str] = {
    "short": "short", "medium": "medium", "long": "long"
}
CATEGORIES: Dict[str, str] = {
    "films": "2", "vehicles": "3", "art": "4", "sports": "5", "travels": "6",
    "gaming": "7", "people": "8", "comedy": "9", "entertainment": "10",
    "news_politics": "11", "howto": "12", "education": "13", "activism": "14",
    "science_technology": "15", "animals": "16", "documentary": "17", "food": "18",
    "music": "1", "other": "19"
}
LANGUAGES: Dict[str, str] = {
    "english": "en", "spanish": "es", "french": "fr", "german": "de", "italian": "it",
    "portuguese": "pt", "russian": "ru", "chinese": "zh", "japanese": "ja", "arabic": "ar",
    "hindi": "hi", "bengali": "bn", "punjabi": "pa", "turkish": "tr", "dutch": "nl",
    "swedish": "sv", "polish": "pl", "ukrainian": "uk", "vietnamese": "vi", "korean": "ko",
    "indonesian": "id", "finnish": "fi", "norwegian": "no", "danish": "da", "czech": "cs",
    "hungarian": "hu", "romanian": "ro", "greek": "el", "thai": "th", "hebrew": "he",
    "persian": "fa", "urdu": "ur", "filipino": "fil", "malay": "ms", "swahili": "sw"
}
SORT_OPTIONS: Dict[str, str] = {
    "match": "-match", "published_at": "-publishedAt", "created_at": "createdAt",
    "views": "-views", "likes": "-likes", "hot": "-hot"
}
SEARCH_TIMEOUT: int = 15
SEARCH_MAX_RETRIES: int = 3
SEARCH_RETRY_DELAY: float = 1.0

# --- Re-export cache sizes for backward compatibility ---
SEARCH_CACHE_SIZE = CacheConfig.SEARCH_CACHE_SIZE
TRENDING_CACHE_SIZE = CacheConfig.TRENDING_CACHE_SIZE
FEED_CACHE_SIZE = CacheConfig.FEED_CACHE_SIZE
CHANNEL_CACHE_SIZE = CacheConfig.CHANNEL_CACHE_SIZE

# --- Validazione dell'Ambiente ---
def validate_environment() -> None:
    try:
        import sqlite3
        import requests
        import feedparser
        import vlc
        import yt_dlp
    except ImportError as e:
        raise RuntimeError(f"Dipendenza richiesta mancante: {e}")
    if not DATA_DIR.exists():
        raise RuntimeError(f"La directory dati non esiste: {DATA_DIR}")
    if not os.access(DATA_DIR, os.W_OK):
        raise RuntimeError(f"Permessi di scrittura insufficienti per: {DATA_DIR}")

validate_environment()

