from PyQt5.QtWidgets import (
    QWidget, QVBoxLayout, QLabel, QListWidget, QListWidgetItem,
    QPushButton, QHBoxLayout, QMenu, QMessageBox, QInputDialog, QGridLayout,
    QFrame, QProgressDialog, QFileDialog, QApplication, QScrollArea, QScrollBar
)
from PyQt5.QtCore import Qt, pyqtSignal, QThreadPool, QObject, QTimer, QPropertyAnimation, QEasingCurve
from PyQt5.QtGui import QPixmap, QPainter
from core.models import Video, Playlist
from core.services.playlist_service import PlaylistService
from core.player import DownloadWorker
from config.theme import Theme
import logging
import threading
import requests
from pathlib import Path

logger = logging.getLogger(__name__)

class InfiniteScrollGrid(QWidget):
    """Widget con griglia a scorrimento infinito."""
    load_more_requested = pyqtSignal(int)  # page number

    def __init__(self, parent=None):
        super().__init__(parent)
        self.current_page = 0
        self.is_loading = False
        self.has_more = True
        self.columns = 2
        self.setup_ui()
        self.setup_scroll_handling()

    def setup_ui(self):
        """Configura l'interfaccia per lo scrolling infinito."""
        main_layout = QVBoxLayout(self)
        main_layout.setContentsMargins(0, 0, 0, 0)
        main_layout.setSpacing(0)

        # Area di scorrimento
        self.scroll_area = QScrollArea()
        self.scroll_area.setWidgetResizable(True)
        self.scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        self.scroll_area.setStyleSheet(f"""
            QScrollArea {{
                border: none;
                background: {Theme.BACKGROUND};
            }}
            QScrollBar:vertical {{
                background: {Theme.SURFACE};
                width: 12px;
                margin: 0px;
                border-radius: 6px;
            }}
            QScrollBar::handle:vertical {{
                background: {Theme.PRIMARY};
                border-radius: 6px;
                min-height: 20px;
            }}
            QScrollBar::handle:vertical:hover {{
                background: {Theme.PRIMARY_HOVER};
            }}
            QScrollBar::add-line:vertical, QScrollBar::sub-line:vertical {{
                border: none;
                background: none;
            }}
        """)

        # Widget contenitore per la griglia
        self.grid_widget = QWidget()
        self.grid_layout = QGridLayout(self.grid_widget)
        self.grid_layout.setSpacing(20)
        self.grid_layout.setContentsMargins(16, 16, 16, 16)
        self.grid_layout.setAlignment(Qt.AlignTop)

        self.scroll_area.setWidget(self.grid_widget)
        main_layout.addWidget(self.scroll_area)

        # Indicatore di caricamento
        self.loading_indicator = QLabel("Caricamento...")
        self.loading_indicator.setAlignment(Qt.AlignCenter)
        self.loading_indicator.setStyleSheet(f"""
            QLabel {{
                color: {Theme.TEXT_SECONDARY};
                font-size: 14px;
                padding: 20px;
                background: {Theme.SURFACE};
                border-radius: 8px;
                margin: 10px;
            }}
        """)
        self.loading_indicator.hide()
        main_layout.addWidget(self.loading_indicator)

    def setup_scroll_handling(self):
        """Configura il rilevamento dello scrolling."""
        self.scroll_timer = QTimer()
        self.scroll_timer.setSingleShot(True)
        self.scroll_timer.timeout.connect(self.check_scroll_position)

        # Connessione per lo scrolling
        scrollbar = self.scroll_area.verticalScrollBar()
        scrollbar.valueChanged.connect(self.on_scroll)

    def on_scroll(self, value):
        """Gestisce l'evento di scrolling."""
        self.scroll_timer.start(100)  # Debounce di 100ms

    def check_scroll_position(self):
        """Controlla se è necessario caricare più contenuti."""
        if self.is_loading or not self.has_more:
            return

        scrollbar = self.scroll_area.verticalScrollBar()
        scroll_position = scrollbar.value()
        max_position = scrollbar.maximum()

        # Carica più contenuti quando si è vicini alla fine
        if scroll_position >= max_position * 0.8:
            self.load_next_page()

    def load_next_page(self):
        """Carica la pagina successiva."""
        if self.is_loading or not self.has_more:
            return

        self.is_loading = True
        self.show_loading_indicator()
        self.current_page += 1
        self.load_more_requested.emit(self.current_page)

    def show_loading_indicator(self):
        """Mostra l'indicatore di caricamento."""
        self.loading_indicator.show()

    def hide_loading_indicator(self):
        """Nasconde l'indicatore di caricamento."""
        self.loading_indicator.hide()

    def add_card(self, card, position):
        """Aggiunge una card alla griglia."""
        row = position // self.columns
        col = position % self.columns
        self.grid_layout.addWidget(card, row, col)

    def clear_cards(self):
        """Rimuove tutte le cards dalla griglia."""
        for i in reversed(range(self.grid_layout.count())):
            widget = self.grid_layout.itemAt(i).widget()
            if widget:
                widget.setParent(None)
        self.current_page = 0
        self.has_more = True
        self.is_loading = False

    def set_loading_state(self, loading, has_more=True):
        """Imposta lo stato di caricamento."""
        self.is_loading = loading
        self.has_more = has_more
        if not loading:
            self.hide_loading_indicator()

class AnimatedPlaylistVideoCard(QFrame):
    """Card video personalizzata con animazioni per la playlist."""
    play_requested = pyqtSignal(object)
    add_to_queue_requested = pyqtSignal(object)
    remove_requested = pyqtSignal(object, int)  # video, playlist_id

    def __init__(self, video: Video, playlist_id: int, main_window=None):
        super().__init__()
        self.video = video
        self.playlist_id = playlist_id
        self.main_window = main_window
        self.thread_pool = QThreadPool.globalInstance()
        self.setup_ui()
        self.setup_animations()

    def setup_ui(self):
        """Configura l'interfaccia della card video."""
        self.setFixedSize(530, 340)
        self.setStyleSheet(Theme.video_card_style())

        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        layout.setSpacing(0)

        # Contenitore per la miniatura
        self.thumb_container = QFrame()
        self.thumb_container.setFixedSize(530, 290)
        self.thumb_container.setStyleSheet(f"""
            QFrame {{
                background: {Theme.BACKGROUND};
                border-top-left-radius: 12px;
                border-top-right-radius: 12px;
                border: none;
            }}
        """)
        thumb_layout = QVBoxLayout(self.thumb_container)
        thumb_layout.setContentsMargins(0, 0, 0, 0)

        self.thumb_label = QLabel(self.thumb_container)
        self.thumb_label.setObjectName("thumb_label")
        self.thumb_label.setFixedSize(530, 280)
        self.thumb_label.setAlignment(Qt.AlignCenter)
        self.thumb_label.setStyleSheet(f"""
            QLabel#thumb_label {{
                background: {Theme.BACKGROUND};
                border-top-left-radius: 12px;
                border-top-right-radius: 12px;
                border: none;
            }}
        """)
        self.thumb_label.setScaledContents(True)
        thumb_layout.addWidget(self.thumb_label)

        # Pulsante di eliminazione (icona X in alto a destra)
        self.delete_btn = QPushButton("×", self.thumb_container)
        self.delete_btn.setFixedSize(40, 40)
        self.delete_btn.setStyleSheet(f"""
            QPushButton {{
                background-color: rgba(255, 0, 0, 0.8);
                color: white;
                font-size: 24px;
                font-weight: bold;
                border-radius: 20px;
                border: 2px solid white;
            }}
            QPushButton:hover {{
                background-color: rgba(255, 0, 0, 1.0);
            }}
        """)
        self.delete_btn.move(480, 10)
        self.delete_btn.hide()
        self.delete_btn.clicked.connect(self.remove_video)

        # Pulsante di riproduzione sovrapposto
        self.play_overlay = QPushButton("▶", self.thumb_container)
        self.play_overlay.setFixedSize(68, 48)
        self.play_overlay.setStyleSheet(f"""
            QPushButton {{
                background-color: rgba(255, 255, 255, 0.9);
                color: black;
                font-size: 20px;
                border-radius: 4px;
                border: none;
            }}
            QPushButton:hover {{
                background-color: white;
            }}
        """)
        self.play_overlay.hide()
        self.play_overlay.clicked.connect(lambda: self.play_requested.emit(self.video))

        # Overlay scuro
        self.overlay = QFrame(self.thumb_container)
        self.overlay.setFixedSize(530, 290)
        self.overlay.setStyleSheet(f"""
            QFrame {{
                background-color: rgba(0, 0, 0, 0.3);
                border-radius: 12px;
            }}
        """)
        self.overlay.hide()

        # Etichetta durata
        self.duration_label = QLabel(self.thumb_container)
        self.duration_label.setObjectName("duration")
        duration_text = f" {self._format_duration(self.video.duration)} " if hasattr(self.video, 'duration') else " 0:00 "
        self.duration_label.setText(duration_text)
        self.duration_label.setStyleSheet(f"""
            QLabel#duration {{
                background-color: rgba(0, 0, 0, 0.8);
                color: white;
                border: none;
                border-radius: 2px;
                padding: 2px 4px;
                font-size: 12px;
                font-weight: 500;
                min-width: 40px;
                text-align: center;
            }}
        """)
        self.duration_label.move(470, 235)

        # Frame per le informazioni
        info_frame = QFrame()
        info_frame.setFixedSize(530, 78)
        info_frame.setStyleSheet(f"""
            QFrame {{
                background: {Theme.SURFACE};
                border: none;
                border-bottom-left-radius: 12px;
                border-bottom-right-radius: 12px;
            }}
        """)
        info_layout = QVBoxLayout(info_frame)
        info_layout.setContentsMargins(12, 8, 12, 8)
        info_layout.setSpacing(4)

        # Titolo
        title_text = self.video.title[:50] + "..." if len(self.video.title) > 50 else self.video.title
        self.title_label = QLabel(title_text)
        self.title_label.setObjectName("title")
        self.title_label.setWordWrap(True)
        self.title_label.setToolTip(self.video.title)
        self.title_label.setStyleSheet(f"""
            QLabel#title {{
                color: {Theme.TEXT_PRIMARY};
                font-size: 14px;
                font-weight: 500;
            }}
        """)
        info_layout.addWidget(self.title_label)

        # Autore
        if hasattr(self.video, 'author') and self.video.author:
            self.author_label = QLabel(f"👤 {self.video.author}")
            self.author_label.setObjectName("channel")
            self.author_label.setStyleSheet(f"""
                QLabel#channel {{
                    color: {Theme.TEXT_SECONDARY};
                    font-size: 12px;
                }}
            """)
            info_layout.addWidget(self.author_label)

        # Data di pubblicazione
        if hasattr(self.video, 'published') and self.video.published:
            pub_date = self.video.published.strftime("%d/%m/%Y")
            self.date_label = QLabel(f"📅 {pub_date}")
            self.date_label.setObjectName("date_label")
            self.date_label.setStyleSheet(f"""
                QLabel#date_label {{
                    color: {Theme.TEXT_SECONDARY};
                    font-size: 12px;
                }}
            """)
            info_layout.addWidget(self.date_label)

        info_layout.addStretch()

        # Aggiungi i contenitori al layout principale
        layout.addWidget(self.thumb_container)
        layout.addWidget(info_frame)

        # Gestione eventi
        self.setMouseTracking(True)
        self.thumb_container.setMouseTracking(True)
        self.thumb_container.installEventFilter(self)
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self.show_context_menu)

        # Carica la miniatura se disponibile
        if hasattr(self.video, 'thumbnail_url') and self.video.thumbnail_url:
            self.load_thumbnail(self.video.thumbnail_url)

    def setup_animations(self):
        """Configura le animazioni per la card."""
        # Animazione per l'ingresso
        self.enter_animation = QPropertyAnimation(self, b"windowOpacity")
        self.enter_animation.setDuration(300)
        self.enter_animation.setStartValue(0.0)
        self.enter_animation.setEndValue(1.0)
        self.enter_animation.setEasingCurve(QEasingCurve.OutCubic)

        # Animazione per l'hover
        self.hover_animation = QPropertyAnimation(self, b"geometry")
        self.hover_animation.setDuration(200)
        self.hover_animation.setEasingCurve(QEasingCurve.OutCubic)

    def enterEvent(self, event):
        """Animazione all'ingresso del mouse."""
        self.on_hover_enter()
        super().enterEvent(event)

    def leaveEvent(self, event):
        """Animazione all'uscita del mouse."""
        self.on_hover_leave()
        super().leaveEvent(event)

    def _format_duration(self, seconds: str) -> str:
        """Formatta la durata in MM:SS o H:MM:SS."""
        try:
            sec = int(seconds)
            h, m, s = sec // 3600, (sec % 3600) // 60, sec % 60
            if h:
                return f"{h}:{m:02d}:{s:02d}"
            return f"{m}:{s:02d}"
        except ValueError:
            return "0:00"

    def load_thumbnail(self, url):
        """Carica la miniatura in modo asincrono."""
        def _load():
            try:
                response = requests.get(url, timeout=10)
                response.raise_for_status()
                pixmap = QPixmap()
                if pixmap.loadFromData(response.content):
                    from PyQt5.QtCore import QMetaObject, Q_ARG
                    QMetaObject.invokeMethod(
                        self.thumb_label, "setPixmap",
                        Qt.QueuedConnection,
                        Q_ARG(QPixmap, pixmap.scaled(
                            self.thumb_label.size(),
                            Qt.KeepAspectRatio,
                            Qt.SmoothTransformation
                        ))
                    )
                else:
                    from PyQt5.QtCore import QMetaObject, Q_ARG
                    QMetaObject.invokeMethod(
                        self.thumb_label, "setText",
                        Qt.QueuedConnection,
                        Q_ARG(str, "Impossibile caricare la miniatura")
                    )
            except Exception as e:
                logger.error(f"Errore nel caricamento della miniatura: {e}")
                from PyQt5.QtCore import QMetaObject, Q_ARG
                QMetaObject.invokeMethod(
                    self.thumb_label, "setText",
                    Qt.QueuedConnection,
                    Q_ARG(str, "Impossibile caricare la miniatura")
                )
        threading.Thread(target=_load, daemon=True).start()

    def eventFilter(self, obj, event):
        """Gestisci gli eventi di hover per il contenitore della miniatura."""
        if obj is self.thumb_container:
            if event.type() == event.Enter:
                self.on_hover_enter()
            elif event.type() == event.Leave:
                self.on_hover_leave()
        return super().eventFilter(obj, event)

    def on_hover_enter(self):
        """Mostra overlay, pulsante play e pulsante di eliminazione al passaggio del mouse."""
        self.overlay.show()
        self.play_overlay.show()
        self.delete_btn.show()
        self.play_overlay.move(
            (self.thumb_container.width() - self.play_overlay.width()) // 2,
            (self.thumb_container.height() - self.play_overlay.height()) // 2
        )
        self.play_overlay.raise_()
        self.overlay.raise_()
        self.delete_btn.raise_()

    def on_hover_leave(self):
        """Nascondi overlay, pulsante play e pulsante di eliminazione quando il mouse esce."""
        self.overlay.hide()
        self.play_overlay.hide()
        self.delete_btn.hide()

    def mousePressEvent(self, event):
        """Gestisci gli eventi di clic per la riproduzione del video."""
        if event.button() == Qt.LeftButton:
            self.play_requested.emit(self.video)
        super().mousePressEvent(event)

    def show_context_menu(self, pos):
        """Mostra il menu contestuale con le opzioni del video."""
        menu = QMenu(self)
        menu.setStyleSheet(f"""
            QMenu {{
                background: {Theme.SURFACE};
                border: 1px solid {Theme.BORDER};
                border-radius: 6px;
                padding: 4px;
                color: {Theme.TEXT_PRIMARY};
            }}
            QMenu::item:selected {{
                background: rgba(106, 90, 205, 0.2);
                border-radius: 4px;
            }}
        """)

        menu.addAction("Riproduci video", lambda: self.play_requested.emit(self.video))
        menu.addAction("Aggiungi alla coda", lambda: self.add_to_queue_requested.emit(self.video))
        menu.addAction("Scarica video", self.download_video)
        menu.addAction("Rimuovi dalla playlist", self.remove_video)
        menu.exec_(self.mapToGlobal(pos))

    def remove_video(self):
        """Rimuovi il video dalla playlist."""
        reply = QMessageBox.question(
            self,
            "Rimuovi video",
            f"Rimuovere '{self.video.title}' da questa playlist?",
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No
        )
        if reply == QMessageBox.Yes:
            self.remove_requested.emit(self.video, self.playlist_id)

    def download_video(self):
        """Scarica il video con tracciamento dello stato."""
        download_dir = QFileDialog.getExistingDirectory(
            self,
            "Seleziona la cartella di destinazione",
            str(Path.home())
        )
        if not download_dir:
            return

        progress_dialog = QProgressDialog(
            f"Scaricamento in corso: {self.video.title}",
            "Annulla",
            0,
            100,
            self
        )
        progress_dialog.setWindowTitle("Scaricamento in corso")
        progress_dialog.setWindowModality(Qt.WindowModal)
        progress_dialog.setAutoClose(False)
        progress_dialog.setValue(0)

        def update_progress(progress: int):
            progress_dialog.setValue(progress)

        def on_completed(filename: str, message: str):
            if not progress_dialog.wasCanceled():
                progress_dialog.setValue(100)
                QMessageBox.information(self, "Successo", f"Video salvato: {filename}")
            progress_dialog.close()

        def on_error(url: str, error: str):
            if not progress_dialog.wasCanceled():
                QMessageBox.warning(self, "Errore", f"Scaricamento fallito: {error}")
            progress_dialog.close()

        worker = DownloadWorker(self.video.url, download_dir)
        worker.signals.progress.connect(update_progress)
        worker.signals.finished.connect(on_completed)
        worker.signals.error.connect(on_error)
        progress_dialog.canceled.connect(worker.stop)
        self.thread_pool.start(worker)
        progress_dialog.exec_()

    def keyPressEvent(self, event):
        """Gestisci le scorciatoie da tastiera."""
        if event.key() == Qt.Key_Enter or event.key() == Qt.Key_Return:
            self.play_requested.emit(self.video)
        elif event.key() == Qt.Key_D and event.modifiers() == Qt.ControlModifier:
            self.download_video()
        elif event.key() == Qt.Key_Delete:
            self.remove_video()
        else:
            super().keyPressEvent(event)

class EnhancedPlaylistPanel(QWidget):
    video_card_play_requested = pyqtSignal(object)
    add_to_queue_requested = pyqtSignal(object)

    def __init__(self, main_window):
        super().__init__()
        self.main_window = main_window
        self.current_playlist_id = None
        self.videos_batch_size = 9  # 3 righe × 3 colonne
        self.setup_ui()
        self.refresh_playlists()

    def setup_ui(self):
        """Inizializza l'interfaccia utente migliorata."""
        layout = QVBoxLayout(self)
        layout.setContentsMargins(16, 16, 16, 16)
        layout.setSpacing(16)

        # Titolo
        title_label = QLabel("Le tue playlist")
        title_label.setStyleSheet(f"""
            color: {Theme.TEXT_PRIMARY}; 
            font-size: 24px; 
            font-weight: bold;
            font-family: {Theme.FONT_FAMILY};
            margin-bottom: 8px;
        """)
        layout.addWidget(title_label)

        # Controlli playlist
        controls_layout = QHBoxLayout()
        controls_layout.setContentsMargins(0, 0, 0, 0)
        controls_layout.setSpacing(12)

        # Pulsante nuova playlist
        self.new_playlist_btn = QPushButton("🎵 Nuova playlist")
        self.new_playlist_btn.setFixedHeight(40)
        self.new_playlist_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.PRIMARY};
                color: white;
                border: none;
                border-radius: 8px;
                padding: 8px 16px;
                font-family: {Theme.FONT_FAMILY};
                font-size: 14px;
                font-weight: 500;
            }}
            QPushButton:hover {{
                background: {Theme.PRIMARY_HOVER};
                transform: translateY(-1px);
            }}
            QPushButton:pressed {{
                background: {Theme.PRIMARY};
                transform: translateY(0px);
            }}
        """)
        self.new_playlist_btn.clicked.connect(self.create_new_playlist)

        # Pulsante elimina playlist
        self.delete_playlist_btn = QPushButton("🗑️ Elimina playlist")
        self.delete_playlist_btn.setFixedHeight(40)
        self.delete_playlist_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.DANGER};
                color: white;
                border: none;
                border-radius: 8px;
                padding: 8px 16px;
                font-family: {Theme.FONT_FAMILY};
                font-size: 14px;
                font-weight: 500;
            }}
            QPushButton:hover {{
                background: #ff5555;
                transform: translateY(-1px);
            }}
            QPushButton:pressed {{
                background: {Theme.DANGER};
                transform: translateY(0px);
            }}
        """)
        self.delete_playlist_btn.clicked.connect(self.delete_selected_playlist)

        # Pulsante riproduci tutto
        self.play_all_btn = QPushButton("🎬 Riproduci tutto")
        self.play_all_btn.setFixedHeight(40)
        self.play_all_btn.setStyleSheet(f"""
            QPushButton {{
                background: {Theme.TERTIARY};
                color: white;
                border: none;
                border-radius: 8px;
                padding: 8px 16px;
                font-family: {Theme.FONT_FAMILY};
                font-size: 14px;
                font-weight: 500;
            }}
            QPushButton:hover {{
                background: {Theme.TERTIARY_HOVER};
                transform: translateY(-1px);
            }}
            QPushButton:pressed {{
                background: {Theme.TERTIARY};
                transform: translateY(0px);
            }}
        """)
        self.play_all_btn.clicked.connect(self.play_all_videos)

        controls_layout.addWidget(self.new_playlist_btn)
        controls_layout.addWidget(self.delete_playlist_btn)
        controls_layout.addWidget(self.play_all_btn)
        controls_layout.addStretch()
        layout.addLayout(controls_layout)

        # Container principale diviso
        split_layout = QHBoxLayout()
        split_layout.setSpacing(20)
        split_layout.setContentsMargins(0, 0, 0, 0)

        # Lista playlist (sidebar sinistra)
        playlist_container = QWidget()
        playlist_container.setFixedWidth(300)
        playlist_layout = QVBoxLayout(playlist_container)
        playlist_layout.setContentsMargins(0, 0, 0, 0)
        playlist_layout.setSpacing(8)

        playlist_label = QLabel("Playlist")
        playlist_label.setStyleSheet(f"""
            color: {Theme.TEXT_PRIMARY};
            font-size: 16px;
            font-weight: 600;
            font-family: {Theme.FONT_FAMILY};
            padding: 8px 4px;
        """)
        playlist_layout.addWidget(playlist_label)

        self.playlist_list = QListWidget()
        self.playlist_list.setStyleSheet(f"""
            QListWidget {{
                background: {Theme.SURFACE};
                border: 1px solid {Theme.BORDER};
                border-radius: 8px;
                color: {Theme.TEXT_PRIMARY};
                font-family: {Theme.FONT_FAMILY};
                font-size: 14px;
                padding: 4px;
            }}
            QListWidget::item {{
                padding: 12px 8px;
                border-radius: 6px;
                margin: 2px;
            }}
            QListWidget::item:selected {{
                background: {Theme.PRIMARY};
                color: white;
            }}
            QListWidget::item:hover {{
                background: {Theme.SURFACE_HOVER};
            }}
        """)
        self.playlist_list.itemClicked.connect(self.show_playlist_videos)
        self.playlist_list.setContextMenuPolicy(Qt.CustomContextMenu)
        self.playlist_list.customContextMenuRequested.connect(self.show_playlist_context_menu)
        playlist_layout.addWidget(self.playlist_list)

        split_layout.addWidget(playlist_container)

        # Area video (destra) con scrolling infinito
        video_container = QWidget()
        video_layout = QVBoxLayout(video_container)
        video_layout.setContentsMargins(0, 0, 0, 0)
        video_layout.setSpacing(12)

        # Header area video
        self.video_header = QLabel("Seleziona una playlist")
        self.video_header.setStyleSheet(f"""
            color: {Theme.TEXT_PRIMARY};
            font-size: 18px;
            font-weight: 600;
            font-family: {Theme.FONT_FAMILY};
            padding: 8px 4px;
        """)
        video_layout.addWidget(self.video_header)

        # Griglia con scrolling infinito
        self.infinite_grid = InfiniteScrollGrid()
        self.infinite_grid.load_more_requested.connect(self.load_more_videos)
        video_layout.addWidget(self.infinite_grid)

        split_layout.addWidget(video_container, stretch=1)
        layout.addLayout(split_layout, stretch=1)

    def refresh_playlists(self):
        """Aggiorna la lista delle playlist."""
        logger.info("📁 Aggiornamento playlist")
        self.playlist_list.clear()
        playlists = PlaylistService.get_playlists()
        for playlist in playlists:
            item = QListWidgetItem(playlist.name)
            item.setData(Qt.UserRole, playlist.id)
            self.playlist_list.addItem(item)
        if self.playlist_list.count() > 0:
            self.playlist_list.setCurrentRow(0)
            self.show_playlist_videos(self.playlist_list.item(0))

    def create_new_playlist(self):
        """Crea una nuova playlist."""
        name, ok = QInputDialog.getText(
            self,
            "Nuova playlist",
            "Inserisci il nome della playlist:",
            text="La mia playlist"
        )
        if ok and name:
            try:
                playlist = Playlist(name=name)
                playlist_id = PlaylistService.save_playlist(playlist)
                self.refresh_playlists()
                for i in range(self.playlist_list.count()):
                    item = self.playlist_list.item(i)
                    if item.data(Qt.UserRole) == playlist_id:
                        self.playlist_list.setCurrentItem(item)
                        break
                self.main_window.statusBar().showMessage(f"Creata nuova playlist: {name}")
            except Exception as e:
                logger.error(f"❌ Errore nella creazione della playlist: {e}")
                QMessageBox.warning(self, "Errore", f"Impossibile creare la playlist: {str(e)}")

    def delete_selected_playlist(self):
        """Elimina la playlist selezionata."""
        current_item = self.playlist_list.currentItem()
        if not current_item:
            QMessageBox.warning(self, "Attenzione", "Nessuna playlist selezionata")
            return

        playlist_name = current_item.text()
        playlist_id = current_item.data(Qt.UserRole)
        if playlist_name == "Guarda più tardi":
            QMessageBox.warning(self, "Attenzione", "Non è possibile eliminare la playlist 'Guarda più tardi'")
            return

        reply = QMessageBox.question(
            self,
            "Elimina playlist",
            f"Sei sicuro di voler eliminare la playlist '{playlist_name}'?",
            QMessageBox.Yes | QMessageBox.No
        )
        if reply == QMessageBox.Yes:
            try:
                if PlaylistService.remove_playlist(playlist_id):
                    self.refresh_playlists()
                    self.main_window.statusBar().showMessage(f"Playlist eliminata: {playlist_name}")
                else:
                    QMessageBox.warning(self, "Errore", f"Impossibile eliminare la playlist: {playlist_name}")
            except Exception as e:
                logger.error(f"❌ Errore nell'eliminazione della playlist: {e}")
                QMessageBox.warning(self, "Errore", f"Impossibile eliminare la playlist: {str(e)}")

    def show_playlist_videos(self, item):
        """Mostra i video della playlist selezionata."""
        if not item:
            return

        playlist_id = item.data(Qt.UserRole)
        playlist_name = item.text()
        self.current_playlist_id = playlist_id
        
        logger.info(f"📁 Mostro i video per la playlist: {playlist_name} (ID: {playlist_id})")

        # Aggiorna l'header
        self.video_header.setText(f"Video in '{playlist_name}'")

        # Resetta la griglia
        self.infinite_grid.clear_cards()
        
        # Carica il primo batch di video
        self.load_video_batch(playlist_id, 0)

    def load_video_batch(self, playlist_id, page):
        """Carica un batch di video per la pagina specificata."""
        try:
            videos = PlaylistService.get_playlist_videos(playlist_id)
            if not videos:
                self.infinite_grid.set_loading_state(False, False)
                return

            # Calcola l'intervallo per questo batch
            start_idx = page * self.videos_batch_size
            end_idx = start_idx + self.videos_batch_size
            batch_videos = videos[start_idx:end_idx]

            # Aggiungi le card per questo batch
            for idx, video in enumerate(batch_videos):
                try:
                    card = AnimatedPlaylistVideoCard(video, playlist_id, self.main_window)
                    card.play_requested.connect(self.video_card_play_requested.emit)
                    card.add_to_queue_requested.connect(self.add_to_queue_requested.emit)
                    card.remove_requested.connect(self.remove_video_from_playlist)

                    position = start_idx + idx
                    self.infinite_grid.add_card(card, position)
                except Exception as e:
                    logger.error(f"❌ Errore nella creazione della PlaylistVideoCard: {e}")

            # Controlla se ci sono più video da caricare
            has_more = end_idx < len(videos)
            self.infinite_grid.set_loading_state(False, has_more)

        except Exception as e:
            logger.error(f"❌ Errore nel caricamento del batch video: {e}")
            self.infinite_grid.set_loading_state(False, False)

    def load_more_videos(self, page):
        """Carica più video quando si raggiunge la fine dello scrolling."""
        if self.current_playlist_id:
            self.load_video_batch(self.current_playlist_id, page)

    def show_playlist_context_menu(self, pos):
        """Mostra il menu contestuale per le playlist."""
        item = self.playlist_list.itemAt(pos)
        if not item:
            return

        menu = QMenu(self)
        rename_action = menu.addAction("Rinomina playlist")
        rename_action.triggered.connect(lambda: self.rename_playlist(item))
        delete_action = menu.addAction("Elimina playlist")
        delete_action.triggered.connect(lambda: self.delete_playlist(item))
        menu.exec_(self.playlist_list.mapToGlobal(pos))

    def rename_playlist(self, item):
        """Rinomina la playlist selezionata."""
        playlist_name = item.text()
        if playlist_name == "Guarda più tardi":
            QMessageBox.warning(self, "Attenzione", "Non è possibile rinominare la playlist 'Guarda più tardi'")
            return

        new_name, ok = QInputDialog.getText(
            self,
            "Rinomina playlist",
            "Inserisci il nuovo nome:",
            text=playlist_name
        )
        if ok and new_name and new_name != playlist_name:
            try:
                playlist_id = item.data(Qt.UserRole)
                if PlaylistService.update_playlist_name(playlist_id, new_name):
                    self.refresh_playlists()
                    self.main_window.statusBar().showMessage(f"Playlist rinominata in: {new_name}")
                else:
                    QMessageBox.warning(self, "Errore", f"Impossibile rinominare la playlist in: {new_name}")
            except Exception as e:
                logger.error(f"❌ Errore nel rinominare la playlist: {e}")
                QMessageBox.warning(self, "Errore", f"Impossibile rinominare la playlist: {str(e)}")

    def delete_playlist(self, item):
        """Elimina la playlist selezionata dal menu contestuale."""
        self.playlist_list.setCurrentItem(item)
        self.delete_selected_playlist()

    def remove_video_from_playlist(self, video, playlist_id):
        """Rimuovi un video dalla playlist."""
        try:
            if PlaylistService.remove_from_playlist(playlist_id, video.video_id):
                # Ricarica i video per la playlist corrente
                current_item = self.playlist_list.currentItem()
                if current_item and current_item.data(Qt.UserRole) == playlist_id:
                    self.infinite_grid.clear_cards()
                    self.load_video_batch(playlist_id, 0)

                self.main_window.statusBar().showMessage(f"Video rimosso: {video.title}")
                logger.info(f"🗑️ Video '{video.title}' rimosso dalla playlist ID: {playlist_id}")
            else:
                QMessageBox.warning(self, "Errore", f"Impossibile rimuovere il video: {video.title}")
                logger.warning(f"⚠️ Impossibile rimuovere il video '{video.title}' dalla playlist ID: {playlist_id}")
        except Exception as e:
            logger.error(f"❌ Errore nella rimozione del video: {e}", exc_info=True)
            QMessageBox.warning(self, "Errore", f"Impossibile rimuovere il video: {str(e)}")

    def play_all_videos(self):
        """Riproduci tutti i video nella playlist selezionata."""
        current_item = self.playlist_list.currentItem()
        if not current_item:
            QMessageBox.warning(self, "Attenzione", "Nessuna playlist selezionata")
            return

        playlist_id = current_item.data(Qt.UserRole)
        videos = PlaylistService.get_playlist_videos(playlist_id)
        if not videos:
            QMessageBox.warning(self, "Attenzione", "Nessun video in questa playlist")
            return

        self.main_window.vlc_player.queue_manager.clear_queue()
        for video in videos:
            self.add_to_queue_requested.emit(video)

        if videos:
            self.video_card_play_requested.emit(videos[0])

        self.main_window.statusBar().showMessage(f"Riproduzione di {len(videos)} video dalla playlist")

# Alias per mantenere la compatibilità
PlaylistPanel = EnhancedPlaylistPanel
