/* $Id: kmo_extract_spec-test.c,v 1.10 2013-10-08 11:18:56 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:56 $
 * $Revision: 1.10 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <string.h>
#include <sys/stat.h>

#include <cpl.h>

#include "kmclipm_constants.h"
#include "kmclipm_functions.h"
#include "kmclipm_math.h"

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

const char      *path_recipe        = "kmo_extract_spec/";

const char      *valid_files[]      = {"v_data_F3I_all.fits",
                                       "v_data_noise_F3I_all.fits",
                                       "v_data_F3I_all_infinite.fits",
                                       "v_mask_F2I.fits",
                                       "gauss_cube.fits",
                                       "gauss_cube_noise.fits"};

const char      *invalid_files[]    = {"i_data_F2D.fits",
                                       "i_mask_F3I.fits"};

/**
    @defgroup kmo_extract_spec   kmo_extract_spec unit tests

    @{
 */

/**
 * @brief test with data and mask
 */
static void test_extract_spec(const char *path_op1,
                              const char *path_op2,
                              int ret_val)
{
    // create sof-file
    const char *sof_path = "test_extract_spec.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh,
             "%s\n%s\n",
             path_op1, path_op2);
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
           "esorex --suppress-prefix=TRUE kmo_extract_spec "
            "--mask_method=\"mask\" %s", sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_extract_spec.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
 * @brief test with data and centre/radius
 */
static void test_extract_spec_integrated(const char *path_op1,
                                         double centre_x,
                                         double centre_y,
                                         double radius,
                                         int ret_val)
{
    // create sof-file
    const char *sof_path = "test_extract_spec_integrated.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   EXTRACT_DATA\n", path_op1);
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_extract_spec "
            "--centre=\"%g,%g\" --radius=%g %s",
            centre_x, centre_y, radius, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_extract_spec.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
 * @brief test with data and optimal
 */
static void test_extract_spec_optimal(const char *path_op1,
                                         int ret_val)
{
    // create sof-file
    const char *sof_path = "test_extract_spec_optimal.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   EXTRACT_DATA\n", path_op1);
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_extract_spec "
            "--mask_method=\"optimal\" %s", sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_extract_spec.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
    @brief
        Generates test data for kmo_extract_spec.

    @param path Path where the generated test date should be saved to.
*/
static int kmo_generate_test_data_extract_spec(const char *path)
{
    char        file_path[512];

    int         nr_sub_keys         = 21;
    char        *sub_keys[nr_sub_keys],
                *sub_vals[nr_sub_keys];
    int         sub_types[nr_sub_keys];

    sub_keys[0] = CRPIX1;
    sub_types[0] = CPL_TYPE_DOUBLE;
    sub_vals[0] = "17.4";

    sub_keys[1] = CRPIX2;
    sub_types[1] = CPL_TYPE_DOUBLE;
    sub_vals[1] = "17.4.0";

    sub_keys[2] = CRPIX3;
    sub_types[2] = CPL_TYPE_DOUBLE;
    sub_vals[2] = "1.0";

    sub_keys[3] = CRVAL1;
    sub_types[3] = CPL_TYPE_DOUBLE;
    sub_vals[3] = "999.0";

    sub_keys[4] = CRVAL2;
    sub_types[4] = CPL_TYPE_DOUBLE;
    sub_vals[4] = "999.0";

    sub_keys[5] = CRVAL3;
    sub_types[5] = CPL_TYPE_DOUBLE;
    sub_vals[5] = "1.9999999";

    sub_keys[6] = CDELT1;
    sub_types[6] = CPL_TYPE_DOUBLE;
    sub_vals[6] = "1.9";

    sub_keys[7] = CDELT2;
    sub_types[7] = CPL_TYPE_DOUBLE;
    sub_vals[7] = "1.9";

    sub_keys[8] = CDELT3;
    sub_types[8] = CPL_TYPE_DOUBLE;
    sub_vals[8] = "0.001";

    sub_keys[9] = CTYPE1;
    sub_types[9] = CPL_TYPE_STRING;
    sub_vals[9] = "RA---TAN";

    sub_keys[10] = CTYPE2;
    sub_types[10] = CPL_TYPE_STRING;
    sub_vals[10] = "DEC--TAN";

    sub_keys[11] = CTYPE3;
    sub_types[11] = CPL_TYPE_STRING;
    sub_vals[11] = "WAVE";

    sub_keys[12] = CD1_1;
    sub_types[12] = CPL_TYPE_DOUBLE;
    sub_vals[12] = "1.9";

    sub_keys[13] = CD1_2;
    sub_types[13] = CPL_TYPE_DOUBLE;
    sub_vals[13] = "0.0";

    sub_keys[14] = CD1_3;
    sub_types[14] = CPL_TYPE_DOUBLE;
    sub_vals[14] = "0.0";

    sub_keys[15] = CD2_1;
    sub_types[15] = CPL_TYPE_DOUBLE;
    sub_vals[15] = "0.0";

    sub_keys[16] = CD2_2;
    sub_types[16] = CPL_TYPE_DOUBLE;
    sub_vals[16] = "1.9";

    sub_keys[17] = CD2_3;
    sub_types[17] = CPL_TYPE_DOUBLE;
    sub_vals[17] = "0.0";

    sub_keys[18] = CD3_1;
    sub_types[18] = CPL_TYPE_DOUBLE;
    sub_vals[18] = "0.0";

    sub_keys[19] = CD3_2;
    sub_types[19] = CPL_TYPE_DOUBLE;
    sub_vals[19] = "0.0";

    sub_keys[20] = CD3_3;
    sub_types[20] = CPL_TYPE_DOUBLE;
    sub_vals[20] = "0.001";

    /* ----- valid test data ----- */
    // F3I
    cpl_test_eq(0,
            kmo_test_create_F3I_data(test_global_size_x, test_global_size_y,
                         test_global_size_z,  FALSE, test_global_nr_frames,
                         kmo_test_cat_strings(file_path, path, valid_files[0]),
                         NULL, NULL, NULL, 0,
                         sub_keys, sub_vals, sub_types, nr_sub_keys));

    cpl_test_eq(0,
            kmo_test_create_F3I_data_noise(test_global_size_x, test_global_size_y,
                        test_global_size_z,  FALSE, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[1]),
                        NULL, NULL, NULL, 0,
                        sub_keys, sub_vals, sub_types, nr_sub_keys));

    cpl_test_eq(0,
            kmo_test_create_F3I_data_infinite(test_global_size_x, test_global_size_y,
                                              test_global_size_z, test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[2]),
                          NULL, NULL, NULL, 0,
                          sub_keys, sub_vals, sub_types, nr_sub_keys));

    cpl_test_eq(0,
                kmo_test_create_F2I_data(test_global_size_x, test_global_size_y, FALSE,
                          test_global_nr_frames,
                          kmo_test_cat_strings(file_path, path, valid_files[3]),
                          NULL, NULL, NULL, 0,
                          sub_keys, sub_vals, sub_types, nr_sub_keys));

    cpl_test_eq(0,
            kmo_test_create_F3I_data(test_global_size_x, test_global_size_y,
                         test_global_size_z,  TRUE, test_global_nr_frames,
                         kmo_test_cat_strings(file_path, path, valid_files[4]),
                         NULL, NULL, NULL, 0,
                         sub_keys, sub_vals, sub_types, nr_sub_keys));

    cpl_test_eq(0,
            kmo_test_create_F3I_data_noise(test_global_size_x, test_global_size_y,
                        test_global_size_z,  TRUE, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[5]),
                        NULL, NULL, NULL, 0,
                        sub_keys, sub_vals, sub_types, nr_sub_keys));

    /* ----- invalid test data ----- */
    // F2D
    cpl_test_eq(0,
                kmo_test_create_F2D_data(test_global_size_x, test_global_size_y, FALSE,
                        kmo_test_cat_strings(file_path, path, invalid_files[0]),
                        NULL, NULL, NULL, 0,
                        sub_keys, sub_vals, sub_types, nr_sub_keys));

    cpl_test_eq(0,
            kmo_test_create_F3I_data(test_global_size_x, test_global_size_y,
                         test_global_size_z,  FALSE, test_global_nr_frames,
                         kmo_test_cat_strings(file_path, path, invalid_files[1]),
                         NULL, NULL, NULL, 0,
                         sub_keys, sub_vals, sub_types, nr_sub_keys));

    return 0;
}


/**
  @brief    Test of kmo_extract_spec recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_extract_spec-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_extract_spec-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol     = 0.01;

    char    test_path[256],
            op1_file[256],
            op2_file[256],
            out_path[256];

    cpl_propertylist *h = NULL;

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_extract_spec(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        // just generate the data
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    FILE *fd = fopen("log_kmo_extract_spec.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    /* ----- valid tests ----- */
    kmo_test_cat_strings(op2_file, test_path, valid_files[3]);

    //
    // F3I, all slices
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    test_extract_spec(op1_file, op2_file, 0);
    cpl_test_abs(kmo_test_esorex_data(EXTRACT_SPEC, 0), 93.9315, tol);
    test_extract_spec_integrated(op1_file, 4.0, 5.0, 3.0, 0);
    cpl_test_abs(kmo_test_esorex_data(EXTRACT_SPEC, 0), 104.1, tol);
    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    test_extract_spec_optimal(op1_file, 0);
    cpl_test_abs(kmo_test_esorex_data(EXTRACT_SPEC, 0), 0.27742, tol);

    strcpy(out_path, EXTRACT_SPEC);
    kmo_strlower(out_path);
    strcat(out_path, ".fits");
    h = kmclipm_propertylist_load(out_path, 1);
    cpl_test_abs(cpl_propertylist_get_double(h, CDELT1), 0.01, tol);
    cpl_propertylist_delete(h); h = NULL;

    //
    // F3I with noise, all slices
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[1]);
    test_extract_spec(op1_file, op2_file, 0);
    cpl_test_abs(kmo_test_esorex_data(EXTRACT_SPEC, 0), 130.561, tol);
    test_extract_spec_integrated(op1_file, 4.0, 5.0, 3.0, 0);
    cpl_test_abs(kmo_test_esorex_data(EXTRACT_SPEC, 0), 140.877, tol);
    kmo_test_cat_strings(op1_file, test_path, valid_files[5]);
    test_extract_spec_optimal(op1_file, 0);
    cpl_test_abs(kmo_test_esorex_data(EXTRACT_SPEC, 0), 0.31184, tol);

    //
    // F3I with infinites and without noise, all slices
    //
    kmo_test_cat_strings(op1_file, test_path, valid_files[2]);
    test_extract_spec(op1_file, op2_file, 0);
    cpl_test_eq(isnan(kmo_test_esorex_data(EXTRACT_SPEC, 0)), FALSE);
    test_extract_spec_integrated(op1_file, 4.0, 5.0, 3.0, 0);
    cpl_test_eq(isnan(kmo_test_esorex_data(EXTRACT_SPEC, 0)), FALSE);
    test_extract_spec_optimal(op1_file, 0);
    cpl_test_abs(kmclipm_is_nan_or_inf(kmo_test_esorex_data(EXTRACT_SPEC, 0)),
                 0., tol);

    /* ----- invalid tests ----- */
    // invalid data (F2D)
    kmo_test_cat_strings(op1_file, test_path, invalid_files[0]);
    test_extract_spec(op1_file, op2_file, -1);
    test_extract_spec_integrated(op1_file, 4.0, 5.0, 3.0, -1);
    test_extract_spec_optimal(op1_file, -1);

    // invalid mask (F3I)
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    kmo_test_cat_strings(op2_file, test_path, invalid_files[1]);
    test_extract_spec(op1_file, op2_file, -1);

    return cpl_test_end(0);
}

/** @} */
