import path from 'path';
import fsp from 'fs/promises';
import { resolveCountry } from './resolveCountry.js';
import { processUrl, getProtocolId } from './processUrl.js';
import { generateMirrorMd } from './markdown.js';
import { generateMirrorlist } from './mirrorlist.js';
const inputFile = process.env['INPUT'] || path.join(process.cwd(), 'mirrors.json');
const fixtureFile = process.env['FIXTURE'] || path.join(process.cwd(), 'mirrors.fixture.json');
const mirrorList = process.env['MIRRORLIST'] || path.join(process.cwd(), 'mirrorlist');
const mirrorMd = process.env['MIRRORMD'] || path.join(process.cwd(), 'mirrors.md');
const mdHeadFile = process.env['MDHEADER'] || path.join(process.cwd(), 'head.md');
export class ArtixMlg {
    mirrorCounter = 0;
    mirrorUrls;
    mirrorProfilesByMirrorName;
    mirrors;
    constructor(options) {
        this.mirrors = options.mirrors || this.setMirrors(options.fixture);
        this.mirrorUrls = options.mirrorUrls || [];
        this.mirrorProfilesByMirrorName = options.mirrorProfilesByMirrorName || {};
    }
    async run() {
        const input = JSON.parse(await fsp.readFile(inputFile, { encoding: 'utf-8' }));
        input.mirrors.forEach(this.processMirrorProfile);
        input.mirrors.forEach(this.updateUpstream);
        await fsp.writeFile(fixtureFile, JSON.stringify(this.composeMirrorFixture(), null, 4));
        await fsp.writeFile(mirrorList, generateMirrorlist(input.mirrors?.filter(m => m.public && m.active && !m.suppress) || []));
        await fsp.writeFile(mirrorMd, await generateMirrorMd(mdHeadFile, this.mirrorProfilesByMirrorName));
    }
    setMirrors(fixture) {
        const mirrors = {};
        fixture.filter(x => x.model === 'mirrors.Mirror').forEach((mirror) => {
            this.mirrorCounter = Math.max(this.mirrorCounter, mirror.pk);
            mirror.fields.active = false;
            mirror.fields.public = false;
            mirror.fields.isos = false;
            mirrors[mirror.fields.name] = mirror;
        });
        return mirrors;
    }
    getMirror(name) {
        return this.mirrors[name] = {
            pk: this.mirrors[name]?.pk || ++this.mirrorCounter,
            model: 'mirrors.Mirror',
            fields: {
                name: name,
                tier: -1,
                upstream: null,
                admin_email: '',
                alternate_email: '',
                public: true,
                active: false,
                isos: false,
                rsync_user: '',
                rsync_password: '',
                bug: null,
                notes: ''
            }
        };
    }
    updateMirror(m, profile, url) {
        m.fields.tier = Math.max(m.fields.tier, profile.tier);
        m.fields.admin_email = profile.admin_email || m.fields.admin_email;
        m.fields.alternate_email = profile.alternate_email || m.fields.alternate_email;
        m.fields.notes = profile.notes || m.fields.notes;
        m.fields.active ||= profile.active;
        m.fields.public &&= profile.public;
        m.fields.isos ||= !!profile.stable_isos || !!profile.weekly_isos;
        if (url.protocol === 'rsync') {
            m.fields.rsync_user = profile.rsync_user || '';
            m.fields.rsync_password = profile.rsync_password || '';
        }
    }
    processMirrorProfile = (m, index) => {
        const url = processUrl(m.url);
        const mirror = this.getMirror(m.force_mirror_name || url.name);
        this.updateMirror(mirror, m, url);
        const mirrorUrl = {
            pk: index + 1,
            model: 'mirrors.MirrorUrl',
            fields: {
                url: url.partial,
                protocol: getProtocolId(url.protocol),
                mirror: mirror.pk,
                country: resolveCountry(m.country)?.iso2 || undefined,
                // populate ip fields with `mirrorresolv`
                has_ipv4: false,
                has_ipv6: false,
                active: m.active
            }
        };
        this.mirrorUrls.push(mirrorUrl);
        this.pushMirrorProfile(m.force_mirror_name || url.name, m);
    };
    updateUpstream = (m) => {
        const url = processUrl(m.url);
        const mirror = this.mirrors[m.force_mirror_name || url.name];
        mirror.fields.upstream = (m.upstream && this.mirrors[m.upstream]?.pk) || mirror.fields.upstream;
    };
    pushMirrorProfile(name, m) {
        const list = this.mirrorProfilesByMirrorName[name] ||= [];
        list.push(m);
    }
    composeMirrorFixture() {
        const fixture = [];
        for (let mirrorName in this.mirrors) {
            fixture.push(this.mirrors[mirrorName]);
        }
        fixture.push.apply(fixture, this.mirrorUrls);
        return fixture;
    }
}
export default ArtixMlg;
//# sourceMappingURL=artix-mlg.js.map