/* Any copyright is dedicated to the Public Domain.
   http://creativecommons.org/publicdomain/zero/1.0/ */

"use strict";

// Test that we can change the display with which we describe a dominator tree
// while the dominator tree is in the middle of being fetched.

const {
  dominatorTreeState,
  viewState,
  labelDisplays,
  treeMapState,
} = require("resource://devtools/client/memory/constants.js");
const {
  setLabelDisplayAndRefresh,
} = require("resource://devtools/client/memory/actions/label-display.js");
const {
  changeView,
} = require("resource://devtools/client/memory/actions/view.js");
const {
  takeSnapshotAndCensus,
} = require("resource://devtools/client/memory/actions/snapshot.js");

add_task(async function () {
  const front = new StubbedMemoryFront();
  const heapWorker = new HeapAnalysesClient();
  await front.attach();
  const store = Store();
  const { getState, dispatch } = store;

  dispatch(changeView(viewState.DOMINATOR_TREE));

  dispatch(takeSnapshotAndCensus(front, heapWorker));
  await waitUntilCensusState(store, s => s.treeMap, [treeMapState.SAVED]);
  ok(
    !getState().snapshots[0].dominatorTree,
    "There shouldn't be a dominator tree model yet since it is not computed " +
      "until we switch to the dominators view."
  );

  // Wait for the dominator tree to start fetching.
  await waitUntilState(
    store,
    state =>
      state.snapshots[0] &&
      state.snapshots[0].dominatorTree &&
      state.snapshots[0].dominatorTree.state === dominatorTreeState.FETCHING
  );

  ok(
    getState().labelDisplay,
    "We have a default display for describing nodes in a dominator tree"
  );
  equal(
    getState().labelDisplay,
    labelDisplays.coarseType,
    "and the default is coarse type"
  );
  equal(
    getState().labelDisplay,
    getState().snapshots[0].dominatorTree.display,
    "and the newly computed dominator tree has that display"
  );

  // Switch to the allocationStack display while we are still fetching the
  // dominator tree.
  dispatch(
    setLabelDisplayAndRefresh(heapWorker, labelDisplays.allocationStack)
  );

  // Wait for the dominator tree to finish being fetched.
  await waitUntilState(
    store,
    state =>
      state.snapshots[0].dominatorTree.state === dominatorTreeState.LOADED
  );

  equal(
    getState().snapshots[0].dominatorTree.display,
    labelDisplays.allocationStack,
    "The new dominator tree's display is allocationStack"
  );
  equal(
    getState().labelDisplay,
    labelDisplays.allocationStack,
    "as is our requested dominator tree display"
  );

  heapWorker.destroy();
  await front.detach();
});
