/*
 * Copyright © 2014 Jon Turney
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <dix-config.h>

#include <X11/X.h>
#include <X11/Xproto.h>
#include <X11/extensions/windowsdristr.h>

#include "dix/request_priv.h"

#include "dixstruct.h"
#include "extnsionst.h"
#include "scrnintstr.h"
#include "swaprep.h"
#include "protocol-versions.h"
#include "windowsdri.h"
#include "glx/dri_helpers.h"

static int WindowsDRIErrorBase = 0;
static int WindowsDRIEventBase = 0;

static void
WindowsDRIResetProc(ExtensionEntry* extEntry)
{
}

static int
ProcWindowsDRIQueryVersion(ClientPtr client)
{
    REQUEST_SIZE_MATCH(xWindowsDRIQueryVersionReq);

    xWindowsDRIQueryVersionReply reply = {
        .majorVersion = SERVER_WINDOWSDRI_MAJOR_VERSION,
        .minorVersion = SERVER_WINDOWSDRI_MINOR_VERSION,
        .patchVersion = SERVER_WINDOWSDRI_PATCH_VERSION,
    };

    if (client->swapped) {
        swaps(&reply.majorVersion);
        swaps(&reply.minorVersion);
        swapl(&reply.patchVersion);
    }

    return X_SEND_REPLY_SIMPLE(client, reply);
}

static int
ProcWindowsDRIQueryDirectRenderingCapable(ClientPtr client)
{
    REQUEST(xWindowsDRIQueryDirectRenderingCapableReq);
    REQUEST_SIZE_MATCH(xWindowsDRIQueryDirectRenderingCapableReq);

    if (client->swapped)
        swapl(&stuff->screen);

    xWindowsDRIQueryDirectRenderingCapableReply reply = {
        .isCapable = client->local &&
                     glxWinGetScreenAiglxIsActive(screenInfo.screens[stuff->screen])
    };

    return X_SEND_REPLY_SIMPLE(client, reply);
}

static int
ProcWindowsDRIQueryDrawable(ClientPtr client)
{
    REQUEST(xWindowsDRIQueryDrawableReq);
    REQUEST_SIZE_MATCH(xWindowsDRIQueryDrawableReq);

    if (client->swapped) {
        swapl(&stuff->screen);
        swapl(&stuff->drawable);
    }

    int rc;

    xWindowsDRIQueryDrawableReply reply = { 0 };
    rc = glxWinQueryDrawable(client, stuff->drawable, &(reply.drawable_type), &(reply.handle));

    if (rc)
        return rc;

    if (client->swapped) {
        swapl(&reply.handle);
        swapl(&reply.drawable_type);
    }

    return X_SEND_REPLY_SIMPLE(client, reply);
}

static int
ProcWindowsDRIFBConfigToPixelFormat(ClientPtr client)
{
    REQUEST(xWindowsDRIFBConfigToPixelFormatReq);
    REQUEST_SIZE_MATCH(xWindowsDRIFBConfigToPixelFormatReq);

    if (client->swapped) {
        swapl(&stuff->screen);
        swapl(&stuff->fbConfigID);
    }

    xWindowsDRIFBConfigToPixelFormatReply reply = {
        .pixelFormatIndex = glxWinFBConfigIDToPixelFormatIndex(stuff->screen, stuff->fbConfigID)
    };

    if (client->swapped) {
        swapl(&reply.pixelFormatIndex);
    }

    return X_SEND_REPLY_SIMPLE(client, reply);
}

/* dispatch */

static int
ProcWindowsDRIDispatch(ClientPtr client)
{
    REQUEST(xReq);

    switch (stuff->data) {
    case X_WindowsDRIQueryVersion:
        return ProcWindowsDRIQueryVersion(client);

    case X_WindowsDRIQueryDirectRenderingCapable:
        return ProcWindowsDRIQueryDirectRenderingCapable(client);
    }

    if (!client->local)
        return WindowsDRIErrorBase + WindowsDRIClientNotLocal;

    switch (stuff->data) {
    case X_WindowsDRIQueryDrawable:
        return ProcWindowsDRIQueryDrawable(client);

    case X_WindowsDRIFBConfigToPixelFormat:
        return ProcWindowsDRIFBConfigToPixelFormat(client);

    default:
        return BadRequest;
    }
}

static void
SNotifyEvent(xWindowsDRINotifyEvent *from,
             xWindowsDRINotifyEvent *to)
{
    to->type = from->type;
    to->kind = from->kind;
    cpswaps(from->sequenceNumber, to->sequenceNumber);
    cpswapl(from->time, to->time);
}

void
WindowsDRIExtensionInit(void)
{
    ExtensionEntry* extEntry;

    if ((extEntry = AddExtension(WINDOWSDRINAME,
                                 WindowsDRINumberEvents,
                                 WindowsDRINumberErrors,
                                 ProcWindowsDRIDispatch,
                                 ProcWindowsDRIDispatch,
                                 WindowsDRIResetProc,
                                 StandardMinorOpcode))) {
        size_t i;
        WindowsDRIErrorBase = extEntry->errorBase;
        WindowsDRIEventBase = extEntry->eventBase;
        for (i = 0; i < WindowsDRINumberEvents; i++)
            EventSwapVector[WindowsDRIEventBase + i] = (EventSwapPtr)SNotifyEvent;
    }
}
