/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.protocol.oidc.utils;

import jakarta.ws.rs.core.Response;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Base64Url;
import org.keycloak.common.util.SecretGenerator;
import org.keycloak.crypto.HashException;
import org.keycloak.events.EventBuilder;
import org.keycloak.jose.jws.crypto.HashUtils;
import org.keycloak.services.CorsErrorResponseException;
import org.keycloak.services.cors.Cors;

public class PkceUtils {
    private static final Logger logger = Logger.getLogger(PkceUtils.class);
    private static final Pattern VALID_CODE_VERIFIER_PATTERN = Pattern.compile("^[0-9a-zA-Z\\-\\.~_]+$");

    public static String generateCodeVerifier() {
        return Base64Url.encode((byte[])SecretGenerator.getInstance().randomBytes(64));
    }

    public static String encodeCodeChallenge(String codeVerifier, String codeChallengeMethod) {
        try {
            switch (codeChallengeMethod) {
                case "S256": {
                    return PkceUtils.generateS256CodeChallenge(codeVerifier);
                }
            }
            return codeVerifier;
        }
        catch (Exception ex) {
            return null;
        }
    }

    public static String generateS256CodeChallenge(String codeVerifier) throws HashException {
        return HashUtils.sha256UrlEncodedHash((String)codeVerifier, (Charset)StandardCharsets.ISO_8859_1);
    }

    public static boolean validateCodeChallenge(String verifier, String codeChallenge, String codeChallengeMethod) {
        try {
            switch (codeChallengeMethod) {
                case "plain": {
                    return verifier.equals(codeChallenge);
                }
                case "S256": {
                    return PkceUtils.generateS256CodeChallenge(verifier).equals(codeChallenge);
                }
            }
            return false;
        }
        catch (Exception ex) {
            return false;
        }
    }

    public static void checkParamsForPkceEnforcedClient(String codeVerifier, String codeChallenge, String codeChallengeMethod, String authUserId, String authUsername, EventBuilder event, Cors cors) {
        if (codeVerifier == null) {
            String errorMessage = "PKCE code verifier not specified";
            event.detail("reason", errorMessage);
            event.error("code_verifier_missing");
            throw new CorsErrorResponseException(cors, "invalid_grant", errorMessage, Response.Status.BAD_REQUEST);
        }
        PkceUtils.verifyCodeVerifier(codeVerifier, codeChallenge, codeChallengeMethod, authUserId, authUsername, event, cors);
    }

    public static void checkParamsForPkceNotEnforcedClient(String codeVerifier, String codeChallenge, String codeChallengeMethod, String authUserId, String authUsername, EventBuilder event, Cors cors) {
        if (codeChallenge != null && codeVerifier == null) {
            String errorMessage = "PKCE code verifier not specified";
            event.detail("reason", errorMessage);
            event.error("code_verifier_missing");
            throw new CorsErrorResponseException(cors, "invalid_grant", errorMessage, Response.Status.BAD_REQUEST);
        }
        if (codeChallenge == null && codeVerifier != null) {
            String errorMessage = "PKCE code verifier specified but challenge not present in authorization";
            event.detail("reason", errorMessage);
            event.error("invalid_code_verifier");
            throw new CorsErrorResponseException(cors, "invalid_grant", errorMessage, Response.Status.BAD_REQUEST);
        }
        if (codeChallenge != null) {
            PkceUtils.verifyCodeVerifier(codeVerifier, codeChallenge, codeChallengeMethod, authUserId, authUsername, event, cors);
        }
    }

    public static void verifyCodeVerifier(String codeVerifier, String codeChallenge, String codeChallengeMethod, String authUserId, String authUsername, EventBuilder event, Cors cors) {
        if (!PkceUtils.isValidPkceCodeVerifier(codeVerifier)) {
            String errorReason = "Invalid code verifier";
            String errorMessage = "PKCE verification failed: " + errorReason;
            event.detail("reason", errorReason);
            event.error("invalid_code_verifier");
            throw new CorsErrorResponseException(cors, "invalid_grant", errorMessage, Response.Status.BAD_REQUEST);
        }
        logger.debugf("PKCE supporting Client, codeVerifier = %s", (Object)codeVerifier);
        String codeVerifierEncoded = codeVerifier;
        try {
            if (codeChallengeMethod != null && codeChallengeMethod.equals("S256")) {
                logger.debugf("PKCE codeChallengeMethod = %s", (Object)codeChallengeMethod);
                codeVerifierEncoded = PkceUtils.generateS256CodeChallenge(codeVerifier);
            } else {
                logger.debug((Object)"PKCE codeChallengeMethod is plain");
                codeVerifierEncoded = codeVerifier;
            }
        }
        catch (Exception nae) {
            String errorReason = "Unsupported algorithm specified";
            String errorMessage = "PKCE verification failed: " + errorReason;
            event.detail("reason", errorReason);
            event.error("pkce_verification_failed");
            throw new CorsErrorResponseException(cors, "invalid_grant", errorMessage, Response.Status.BAD_REQUEST);
        }
        if (!codeChallenge.equals(codeVerifierEncoded)) {
            String errorReason = "Code mismatch";
            String errorMessage = "PKCE verification failed: " + errorReason;
            event.detail("reason", errorReason);
            event.error("pkce_verification_failed");
            throw new CorsErrorResponseException(cors, "invalid_grant", errorMessage, Response.Status.BAD_REQUEST);
        }
        logger.debugf("PKCE verification success. codeVerifierEncoded = %s, codeChallenge = %s", (Object)codeVerifierEncoded, (Object)codeChallenge);
    }

    private static boolean isValidPkceCodeVerifier(String codeVerifier) {
        if (codeVerifier.length() < 43) {
            logger.debugf(" Error: PKCE codeVerifier length under lower limit , codeVerifier = %s", (Object)codeVerifier);
            return false;
        }
        if (codeVerifier.length() > 128) {
            logger.debugf(" Error: PKCE codeVerifier length over upper limit , codeVerifier = %s", (Object)codeVerifier);
            return false;
        }
        Matcher m = VALID_CODE_VERIFIER_PATTERN.matcher(codeVerifier);
        return m.matches();
    }
}

