﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lex-models/LexModelBuildingServiceRequest.h>
#include <aws/lex-models/LexModelBuildingService_EXPORTS.h>
#include <aws/lex-models/model/ExportType.h>
#include <aws/lex-models/model/ResourceType.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace LexModelBuildingService {
namespace Model {

/**
 */
class GetExportRequest : public LexModelBuildingServiceRequest {
 public:
  AWS_LEXMODELBUILDINGSERVICE_API GetExportRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetExport"; }

  AWS_LEXMODELBUILDINGSERVICE_API Aws::String SerializePayload() const override;

  AWS_LEXMODELBUILDINGSERVICE_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of the bot to export.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  GetExportRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the bot to export.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  GetExportRequest& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of resource to export. </p>
   */
  inline ResourceType GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  inline void SetResourceType(ResourceType value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = value;
  }
  inline GetExportRequest& WithResourceType(ResourceType value) {
    SetResourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The format of the exported data.</p>
   */
  inline ExportType GetExportType() const { return m_exportType; }
  inline bool ExportTypeHasBeenSet() const { return m_exportTypeHasBeenSet; }
  inline void SetExportType(ExportType value) {
    m_exportTypeHasBeenSet = true;
    m_exportType = value;
  }
  inline GetExportRequest& WithExportType(ExportType value) {
    SetExportType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_version;

  ResourceType m_resourceType{ResourceType::NOT_SET};

  ExportType m_exportType{ExportType::NOT_SET};
  bool m_nameHasBeenSet = false;
  bool m_versionHasBeenSet = false;
  bool m_resourceTypeHasBeenSet = false;
  bool m_exportTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace LexModelBuildingService
}  // namespace Aws
